"""
Created by: Claude Code
Session ID: TBD
Date: 2025-11-17
Purpose: Direct MCP client for connecting to Improvado MCP server without Claude Code
"""

import os
import json
import base64
import requests
from typing import Dict, Any, Optional, List
from dataclasses import dataclass
from pathlib import Path

# Import MCP server URL configuration
from config import DEFAULT_MCP_SERVER_URL


# Auto-load .env files if python-dotenv is available
try:
    from dotenv import load_dotenv

    # ALWAYS load chrome-extension-tcs .env first (base credentials)
    # This makes the skill work globally from any location
    default_env = Path('$PROJECT_ROOT/.env')
    if default_env.exists():
        load_dotenv(default_env)

    # Also try relative path (for backward compatibility when run from project)
    env_path = Path(__file__).parent.parent.parent.parent.parent / '.env'
    if env_path.exists() and env_path.resolve() != default_env.resolve():
        load_dotenv(env_path, override=True)

    # Try .env.local last (for local overrides)
    env_local = Path(__file__).parent.parent.parent.parent.parent / '.env.local'
    if env_local.exists():
        load_dotenv(env_local, override=True)

except ImportError:
    # python-dotenv not installed, will use existing env vars
    pass


@dataclass
class MCPServerConfig:
    """Configuration for MCP server connection"""
    url: str
    email: str
    password: str

    @property
    def auth_header(self) -> str:
        """Generate Basic Auth header value"""
        credentials = f"{self.email}:{self.password}"
        encoded = base64.b64encode(credentials.encode()).decode()
        return f"Basic {encoded}"


class ImprovadoMCPClient:
    """
    Direct Python client for Improvado MCP server.

    Connects to Improvado's internal MCP server via HTTP without requiring Claude Code.
    Implements JSON-RPC 2.0 protocol for MCP communication.

    Example:
        client = ImprovadoMCPClient(
            url=DEFAULT_MCP_SERVER_URL,
            email="daniel@improvado.io",
            password="your_password"
        )

        # Initialize connection
        client.initialize()

        # List available tools
        tools = client.list_tools()

        # Call a tool
        result = client.call_tool("clickhouseTool", {
            "query": "SELECT * FROM internal_analytics.src_dts_dsas_extraction_agency LIMIT 5"
        })
    """

    def __init__(
        self,
        url: str = DEFAULT_MCP_SERVER_URL,
        email: Optional[str] = None,
        password: Optional[str] = None
    ):
        """
        Initialize MCP client.

        Args:
            url: MCP server endpoint URL
            email: User email for Basic Auth (defaults to env var DTS_SERVICE_USER)
            password: User password for Basic Auth (defaults to env var DTS_SERVICE_PASSWORD)
        """
        self.config = MCPServerConfig(
            url=url,
            email=email or os.environ.get("DTS_SERVICE_USER", ""),
            password=password or os.environ.get("DTS_SERVICE_PASSWORD", "")
        )
        self.request_id = 0
        self.session = requests.Session()
        self.session.headers.update({
            "Authorization": self.config.auth_header,
            "Content-Type": "application/json"
        })

    def _make_request(self, method: str, params: Optional[Dict[str, Any]] = None) -> Dict[str, Any]:
        """
        Make JSON-RPC 2.0 request to MCP server.

        Args:
            method: JSON-RPC method name (e.g., "initialize", "tools/list", "tools/call")
            params: Method parameters

        Returns:
            JSON-RPC response result

        Raises:
            Exception: If JSON-RPC error occurs or HTTP request fails
        """
        self.request_id += 1

        payload = {
            "jsonrpc": "2.0",
            "method": method,
            "params": params or {},
            "id": self.request_id
        }

        print(f"[MCP Client] Request: {method}")
        if params:
            print(f"[MCP Client] Params: {json.dumps(params, indent=2)}")

        response = self.session.post(self.config.url, json=payload)
        response.raise_for_status()

        data = response.json()

        # Check for JSON-RPC error
        if "error" in data:
            error = data["error"]
            raise Exception(f"MCP Error [{error.get('code')}]: {error.get('message')}")

        return data.get("result", {})

    def initialize(self) -> Dict[str, Any]:
        """
        Initialize MCP connection.

        Returns server capabilities and instructions.

        Returns:
            {
                "protocolVersion": "2025-03-26",
                "capabilities": {...},
                "serverInfo": {...},
                "instructions": "..."
            }
        """
        result = self._make_request("initialize", {
            "protocolVersion": "2025-03-26",
            "capabilities": {"roots": {}},
            "clientInfo": {
                "name": "python-mcp-client",
                "version": "1.0.0"
            }
        })

        print(f"[MCP Client] Server: {result['serverInfo']['name']} v{result['serverInfo']['version']}")
        print(f"[MCP Client] Protocol: {result['protocolVersion']}")

        # Send initialized notification
        self._make_request("initialized")

        return result

    def list_tools(self) -> List[Dict[str, Any]]:
        """
        List all available tools from MCP server.

        Returns:
            List of tool definitions with name, description, and input schema

        Example:
            [
                {
                    "name": "clickhouseTool",
                    "description": "Execute ClickHouse queries",
                    "inputSchema": {...}
                },
                ...
            ]
        """
        result = self._make_request("tools/list")
        tools = result.get("tools", [])

        print(f"[MCP Client] Available tools: {len(tools)}")
        for tool in tools:
            print(f"  - {tool['name']}: {tool['description'][:60]}...")

        return tools

    def call_tool(self, tool_name: str, arguments: Dict[str, Any]) -> Any:
        """
        Call a specific tool on the MCP server.

        Args:
            tool_name: Name of the tool to call
            arguments: Tool-specific arguments

        Returns:
            Tool execution result

        Example:
            result = client.call_tool("clickhouseTool", {
                "query": "SELECT COUNT(*) FROM internal_analytics.src_dts_dsas_extraction_agency"
            })
        """
        result = self._make_request("tools/call", {
            "name": tool_name,
            "arguments": arguments
        })

        return result

    # Convenience methods for specific tools

    def clickhouse_query(self, query: str, parse_result: bool = True) -> Dict[str, Any]:
        """
        Execute ClickHouse query via MCP.

        Args:
            query: SQL query to execute
            parse_result: If True, parse and return data directly. If False, return raw MCP response.

        Returns:
            Query results (parsed if parse_result=True, otherwise raw MCP response)
        """
        result = self.call_tool("clickhouseTool", {"query": query})

        if parse_result and result.get("content"):
            # Parse the nested JSON response
            import json
            text = result["content"][0]["text"]
            parsed = json.loads(text)
            return parsed  # Returns {"success": true, "data": [...], "columns": [...]}

        return result

    def web_search(self, query: str, max_results: int = 5) -> Dict[str, Any]:
        """
        Perform web search via MCP.

        Args:
            query: Search query
            max_results: Maximum number of results

        Returns:
            Search results
        """
        return self.call_tool("webSearchTool", {
            "query": query,
            "maxResults": max_results
        })

    def get_current_workspace_context(self) -> Dict[str, Any]:
        """
        Get current workspace impersonation context.

        Returns:
            Current workspace context or null if not set
        """
        return self.call_tool("getCurrentWorkspaceContextTool", {})

    def switch_workspace_context(
        self,
        rtbm_agency_id: int,
        workspace_id: Optional[int] = None
    ) -> Dict[str, Any]:
        """
        Switch to a different workspace context.

        Args:
            rtbm_agency_id: Agency ID to impersonate
            workspace_id: Optional specific workspace ID

        Returns:
            New workspace context
        """
        args = {"rtbm_agency_id": rtbm_agency_id}
        if workspace_id is not None:
            args["workspace_id"] = workspace_id

        return self.call_tool("switchWorkspaceContextTool", args)

    def discovery_api_request(
        self,
        data_source: str,
        connector_id: str,
        request: Dict[str, Any]
    ) -> Dict[str, Any]:
        """
        Make Discovery API request.

        Args:
            data_source: Data source name (e.g., "facebook", "google_ads")
            connector_id: Connector ID
            request: API request details with method, url, etc.

        Returns:
            Discovery API response
        """
        return self.call_tool("discoveryRequestTool", {
            "dataSource": data_source,
            "connectorId": connector_id,
            "request": request
        })


def main():
    """Improvado MCP CLI - Dynamic command-line interface for MCP server"""
    import argparse
    import sys
    from session import SessionManager

    parser = argparse.ArgumentParser(
        description="Improvado MCP CLI - Access 79+ MCP tools from command line",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  python3 mcp_client.py init                           # Initialize and authenticate
  python3 mcp_client.py session                        # Show current session
  python3 mcp_client.py switch --agency 7661           # Switch workspace context
  python3 mcp_client.py list-tools                     # List all available tools
  python3 mcp_client.py query "SELECT * FROM table"    # Run ClickHouse query
  python3 mcp_client.py call clickhouseTool --query "SELECT 1"
        """
    )

    subparsers = parser.add_subparsers(dest='command', help='Available commands')

    # Init command
    init_parser = subparsers.add_parser('init', help='Initialize MCP client and authenticate')
    init_parser.add_argument('--email', help='User email (defaults to DTS_SERVICE_USER env var)')
    init_parser.add_argument('--url', help=f'MCP server URL (defaults to {DEFAULT_MCP_SERVER_URL})')

    # Session command
    session_parser = subparsers.add_parser('session', help='Manage session state')
    session_parser.add_argument('action', nargs='?', default='show',
                               choices=['show', 'clear', 'info'],
                               help='Session action (default: show)')

    # Switch command
    switch_parser = subparsers.add_parser('switch', help='Switch workspace context')
    switch_parser.add_argument('--agency', type=int, required=True,
                              help='Agency ID (rtbm_agency_id)')
    switch_parser.add_argument('--workspace', type=int,
                              help='Workspace ID (optional)')

    # List-tools command
    list_parser = subparsers.add_parser('list-tools', help='List available MCP tools')
    list_parser.add_argument('--refresh', action='store_true',
                            help='Refresh tools cache from server')
    list_parser.add_argument('--search', help='Search tools by keyword')
    list_parser.add_argument('--category', action='store_true',
                            help='Group tools by category')

    # Call command - disable auto help to handle --help ourselves
    call_parser = subparsers.add_parser('call', help='Call any MCP tool dynamically', add_help=False)
    call_parser.add_argument('tool_name', nargs='?', help='Name of the tool to call')
    call_parser.add_argument('--args', help='JSON string of arguments')
    call_parser.add_argument('-h', '--help', action='store_true', dest='show_help', help='Show tool help')
    # Dynamic arguments will be parsed after tool schema is fetched

    # Query command (shortcut for clickhouseTool)
    query_parser = subparsers.add_parser('query', help='Execute ClickHouse query')
    query_parser.add_argument('sql', help='SQL query to execute')
    query_parser.add_argument('--format', choices=['json', 'table', 'csv'],
                             default='table', help='Output format')

    args, unknown_args = parser.parse_known_args()

    if not args.command:
        parser.print_help()
        sys.exit(1)

    # Initialize session manager
    session = SessionManager()

    try:
        # Command handlers
        if args.command == 'init':
            cmd_init(session, args)
        elif args.command == 'session':
            cmd_session(session, args)
        elif args.command == 'switch':
            cmd_switch(session, args)
        elif args.command == 'list-tools':
            cmd_list_tools(session, args)
        elif args.command == 'call':
            cmd_call(session, args, unknown_args)
        elif args.command == 'query':
            cmd_query(session, args)
        else:
            parser.print_help()
            sys.exit(1)

    except Exception as e:
        print(f"Error: {e}", file=sys.stderr)
        sys.exit(1)


# Command handlers

def show_tool_help(session, tool_name: str):
    """Display detailed help for a specific tool"""
    from tool_registry import ToolRegistry

    tools = session.get_tools_cache()
    if not tools:
        print("No tools cached. Run 'python3 mcp_client.py init' first.")
        return

    registry = ToolRegistry()
    registry.load_tools(tools)
    print(registry.generate_help(tool_name))


def cmd_init(session, args):
    """Initialize MCP client and save session"""
    print("=== Initializing Improvado MCP Client ===\n")

    # Get credentials
    email = args.email or os.environ.get("DTS_SERVICE_USER")
    password = os.environ.get("DTS_SERVICE_PASSWORD")
    url = args.url or DEFAULT_MCP_SERVER_URL

    if not email or not password:
        print("Error: Missing credentials. Set DTS_SERVICE_USER and DTS_SERVICE_PASSWORD environment variables.")
        print("Or use --email flag and ensure DTS_SERVICE_PASSWORD is set.")
        return

    # Create client
    client = ImprovadoMCPClient(url=url, email=email, password=password)

    print(f"Connecting to: {url}")
    print(f"Email: {email}\n")

    # Initialize connection
    server_info = client.initialize()
    print(f"✓ Connected to {server_info['serverInfo']['name']} v{server_info['serverInfo']['version']}")
    print(f"✓ Protocol: {server_info['protocolVersion']}\n")

    # Fetch and cache tools
    print("Fetching available tools...")
    tools = client.list_tools()
    print(f"✓ Found {len(tools)} tools\n")

    # Save session
    session.set_auth(email=email, server_url=url)
    session.set_protocol_version(server_info['protocolVersion'])
    session.set_tools_cache(tools)
    session.save()

    print(f"✓ Session saved to {session.SESSION_FILE}")
    print("\nNext steps:")
    print("  - Run 'python3 mcp_client.py session' to view current state")
    print("  - Run 'python3 mcp_client.py list-tools' to see available tools")
    print("  - Run 'python3 mcp_client.py switch --agency <id>' to set workspace context")


def cmd_session(session, args):
    """Show or manage session state"""
    if args.action == 'show':
        if not session.exists():
            print("No session found. Run 'python3 mcp_client.py init' first.")
            return
        print("=== Current Session ===\n")
        print(session.format_context_display())

    elif args.action == 'clear':
        session.clear()
        print("✓ Session cleared")

    elif args.action == 'info':
        if not session.exists():
            print("No session found.")
            return
        print("=== Session Info ===\n")
        print(f"Session file: {session.SESSION_FILE}")
        print(f"File size: {session.SESSION_FILE.stat().st_size} bytes")
        print()
        print(session.format_context_display())


def cmd_switch(session, args):
    """Switch workspace context"""
    if not session.exists():
        print("Error: No session found. Run 'python3 mcp_client.py init' first.")
        return

    # Get credentials and create client
    auth = session.get_auth()
    password = os.environ.get("DTS_SERVICE_PASSWORD")

    if not password:
        print("Error: DTS_SERVICE_PASSWORD environment variable not set")
        return

    client = ImprovadoMCPClient(url=auth['server_url'], email=auth['email'], password=password)
    client.initialize()

    print(f"Switching to agency {args.agency}...")

    # Switch context
    switch_args = {"rtbm_agency_id": args.agency}
    if args.workspace:
        switch_args["workspace_id"] = args.workspace

    result = client.switch_workspace_context(**switch_args)

    # Parse result and save to session
    if result.get("content"):
        content_text = result["content"][0]["text"]
        context_data = json.loads(content_text)

        session.set_workspace_context(
            rtbm_agency_id=args.agency,
            agency_name=context_data.get("agency_name"),
            workspace_id=context_data.get("workspace_id"),
            workspace_name=context_data.get("workspace_name")
        )

        print(f"✓ Switched to agency {args.agency}")
        if context_data.get("agency_name"):
            print(f"  Agency: {context_data['agency_name']}")
        if context_data.get("workspace_name"):
            print(f"  Workspace: {context_data['workspace_name']}")
    else:
        session.set_workspace_context(rtbm_agency_id=args.agency)
        print(f"✓ Switched to agency {args.agency}")


def cmd_list_tools(session, args):
    """List available MCP tools"""
    from tool_registry import ToolRegistry

    registry = ToolRegistry()

    # Get tools from cache or fetch from server
    tools = None
    if not args.refresh:
        tools = session.get_tools_cache()

    if tools is None:
        # Fetch from server
        if not session.exists():
            print("Error: No session found. Run 'python3 mcp_client.py init' first.")
            return

        auth = session.get_auth()
        password = os.environ.get("DTS_SERVICE_PASSWORD")

        if not password:
            print("Error: DTS_SERVICE_PASSWORD environment variable not set")
            return

        print("Fetching tools from server...")
        client = ImprovadoMCPClient(url=auth['server_url'], email=auth['email'], password=password)
        client.initialize()
        tools = client.list_tools()

        # Update cache
        session.set_tools_cache(tools)
        print()

    registry.load_tools(tools)

    # Search or list all
    if args.search:
        results = registry.search_tools(args.search)
        print(f"=== Search results for '{args.search}' ===\n")
        print(registry.format_tools_table(results))
    elif args.category:
        print(registry.format_tools_by_category())
    else:
        print(registry.format_tools_table())


def cmd_call(session, args, unknown_args):
    """Call any MCP tool dynamically"""
    # Check if --help/-h is requested for tool
    if args.show_help or '--help' in unknown_args or '-h' in unknown_args:
        if not args.tool_name:
            # No tool specified, show generic help
            print("Usage: mcp_client.py call <tool_name> [--args JSON | --key value ...]")
            print("\nUse 'mcp_client.py call <tool_name> --help' to see tool arguments")
            return

        # Show tool-specific help
        show_tool_help(session, args.tool_name)
        return

    if not session.exists():
        print("Error: No session found. Run 'python3 mcp_client.py init' first.")
        return

    # Get client
    auth = session.get_auth()
    password = os.environ.get("DTS_SERVICE_PASSWORD")

    if not password:
        print("Error: DTS_SERVICE_PASSWORD environment variable not set")
        return

    client = ImprovadoMCPClient(url=auth['server_url'], email=auth['email'], password=password)
    client.initialize()

    # Parse arguments
    arguments = {}
    if args.args:
        arguments = json.loads(args.args)
    else:
        # Parse unknown_args as --key value pairs
        i = 0
        while i < len(unknown_args):
            if unknown_args[i].startswith('--'):
                key = unknown_args[i][2:]  # Remove --
                if i + 1 < len(unknown_args) and not unknown_args[i + 1].startswith('--'):
                    value = unknown_args[i + 1]
                    # Try to parse as JSON, otherwise use as string
                    try:
                        arguments[key] = json.loads(value)
                    except (json.JSONDecodeError, ValueError):
                        arguments[key] = value
                    i += 2
                else:
                    arguments[key] = True
                    i += 1
            else:
                i += 1

    print(f"Calling tool: {args.tool_name}")
    print(f"Arguments: {json.dumps(arguments, indent=2)}\n")

    # Call tool
    result = client.call_tool(args.tool_name, arguments)

    # Format output
    print("=== Result ===")
    print(json.dumps(result, indent=2))


def cmd_query(session, args):
    """Execute ClickHouse query (shortcut for clickhouseTool)"""
    if not session.exists():
        print("Error: No session found. Run 'python3 mcp_client.py init' first.")
        return

    # Get client
    auth = session.get_auth()
    password = os.environ.get("DTS_SERVICE_PASSWORD")

    if not password:
        print("Error: DTS_SERVICE_PASSWORD environment variable not set")
        return

    client = ImprovadoMCPClient(url=auth['server_url'], email=auth['email'], password=password)
    client.initialize()

    print("Executing query...\n")

    # Execute query
    result = client.clickhouse_query(args.sql, parse_result=True)

    # Format output
    if args.format == 'json':
        print(json.dumps(result, indent=2))
    elif args.format == 'table':
        if result.get('success') and result.get('data'):
            data = result['data']
            if data:
                # Print table
                columns = result.get('columns', list(data[0].keys()))

                # Calculate column widths
                col_widths = {col: len(str(col)) for col in columns}
                for row in data:
                    for col in columns:
                        col_widths[col] = max(col_widths[col], len(str(row.get(col, ''))))

                # Print header
                header = " | ".join(f"{col:<{col_widths[col]}}" for col in columns)
                print(header)
                print("-" * len(header))

                # Print rows
                for row in data:
                    print(" | ".join(f"{str(row.get(col, '')):<{col_widths[col]}}" for col in columns))

                print(f"\n{len(data)} rows")
            else:
                print("No data returned")
        else:
            print(f"Query failed: {result.get('error', 'Unknown error')}")
    elif args.format == 'csv':
        if result.get('success') and result.get('data'):
            data = result['data']
            if data:
                columns = result.get('columns', list(data[0].keys()))
                print(",".join(columns))
                for row in data:
                    print(",".join(str(row.get(col, '')) for col in columns))
        else:
            print(f"Query failed: {result.get('error', 'Unknown error')}")


if __name__ == "__main__":
    main()
