"""
Created by: Claude Code
Session ID: TBD
Date: 2025-11-24
Purpose: Session management for Improvado MCP CLI - stores auth and workspace context
"""

import json
import os
from pathlib import Path
from typing import Dict, Any, Optional
from datetime import datetime, timezone

# Import MCP server URL configuration
from config import DEFAULT_MCP_SERVER_URL

# Auto-load .env files if python-dotenv is available
try:
    from dotenv import load_dotenv
    # Try to find .env file in project root
    env_path = Path(__file__).parent.parent.parent / '.env'
    if env_path.exists():
        load_dotenv(env_path)
    # Also try .env.local
    env_local = Path(__file__).parent.parent.parent / '.env.local'
    if env_local.exists():
        load_dotenv(env_local)
except ImportError:
    # python-dotenv not installed, will use existing env vars
    pass


class SessionManager:
    """
    Manages persistent session state for Improvado MCP CLI.

    Stores authentication credentials, workspace context, and cached tools
    in ~/.improvado_mcp_session JSON file.

    Session file format:
    {
        "auth": {
            "email": "user@improvado.io",
            "server_url": "<from IMPROVADO_MCP_SERVER_URL env>"
        },
        "workspace_context": {
            "rtbm_agency_id": 7661,
            "agency_name": "ExampleClient",
            "workspace_id": 123,
            "workspace_name": "Production",
            "switched_at": "2025-11-24T10:30:00Z"
        },
        "tools_cache": [...],
        "tools_cached_at": "2025-11-24T10:00:00Z",
        "protocol_version": "2025-03-26",
        "last_used": "2025-11-24T10:35:00Z"
    }
    """

    SESSION_FILE = Path.home() / ".improvado_mcp_session"
    CACHE_TTL_MINUTES = 5  # Tools cache expires after 5 minutes

    def __init__(self):
        self.session_data: Dict[str, Any] = {}
        self.load()

    def load(self) -> bool:
        """
        Load session from file.

        Returns:
            True if session loaded successfully, False if file doesn't exist
        """
        if not self.SESSION_FILE.exists():
            return False

        try:
            with open(self.SESSION_FILE, 'r') as f:
                self.session_data = json.load(f)
            return True
        except (json.JSONDecodeError, IOError) as e:
            print(f"Warning: Failed to load session file: {e}")
            return False

    def save(self):
        """Save session to file."""
        # Update last_used timestamp
        self.session_data["last_used"] = datetime.now(timezone.utc).isoformat()

        try:
            with open(self.SESSION_FILE, 'w') as f:
                json.dump(self.session_data, f, indent=2)
        except IOError as e:
            print(f"Error: Failed to save session file: {e}")

    def clear(self):
        """Clear session file and in-memory data."""
        self.session_data = {}
        if self.SESSION_FILE.exists():
            self.SESSION_FILE.unlink()

    def exists(self) -> bool:
        """Check if session file exists."""
        return self.SESSION_FILE.exists()

    # Auth management

    def set_auth(self, email: Optional[str] = None, server_url: Optional[str] = None):
        """
        Set authentication credentials.

        Args:
            email: User email (defaults to DTS_SERVICE_USER env var)
            server_url: MCP server URL (defaults to IMPROVADO_MCP_SERVER_URL env var or localhost:3000)

        Note:
            Password is NOT stored in session file for security.
            It should be retrieved from .env or user input each time.
        """
        # Auto-load from environment variables if not provided
        if email is None:
            email = os.environ.get("DTS_SERVICE_USER", "")
        if server_url is None:
            server_url = DEFAULT_MCP_SERVER_URL

        self.session_data["auth"] = {
            "email": email,
            "server_url": server_url
        }

    def get_auth(self) -> Optional[Dict[str, str]]:
        """Get authentication info (email and server_url only)."""
        return self.session_data.get("auth")

    # Workspace context management

    def set_workspace_context(
        self,
        rtbm_agency_id: int,
        agency_name: Optional[str] = None,
        workspace_id: Optional[int] = None,
        workspace_name: Optional[str] = None
    ):
        """
        Set workspace context.

        Args:
            rtbm_agency_id: Agency ID
            agency_name: Agency name (optional)
            workspace_id: Workspace ID (optional)
            workspace_name: Workspace name (optional)
        """
        # Clear tools cache - tools list depends on workspace context
        self.clear_tools_cache()

        context = {
            "rtbm_agency_id": rtbm_agency_id,
            "switched_at": datetime.now(timezone.utc).isoformat()
        }

        if agency_name:
            context["agency_name"] = agency_name
        if workspace_id:
            context["workspace_id"] = workspace_id
        if workspace_name:
            context["workspace_name"] = workspace_name

        self.session_data["workspace_context"] = context
        self.save()

    def get_workspace_context(self) -> Optional[Dict[str, Any]]:
        """Get current workspace context."""
        return self.session_data.get("workspace_context")

    def clear_workspace_context(self):
        """Clear workspace context."""
        if "workspace_context" in self.session_data:
            del self.session_data["workspace_context"]
            self.save()

    # Tools cache management

    def set_tools_cache(self, tools: list):
        """
        Cache tools list from MCP server.

        Args:
            tools: List of tool definitions from list_tools()
        """
        self.session_data["tools_cache"] = tools
        self.session_data["tools_cached_at"] = datetime.now(timezone.utc).isoformat()
        self.save()

    def get_tools_cache(self) -> Optional[list]:
        """
        Get cached tools list if not expired.

        Returns:
            List of tools or None if cache expired or doesn't exist
        """
        if "tools_cache" not in self.session_data:
            return None

        # Check cache expiration
        cached_at_str = self.session_data.get("tools_cached_at")
        if not cached_at_str:
            return None

        try:
            cached_at = datetime.fromisoformat(cached_at_str)
            now = datetime.now(timezone.utc)
            minutes_since_cache = (now - cached_at).total_seconds() / 60

            if minutes_since_cache > self.CACHE_TTL_MINUTES:
                return None  # Cache expired

            return self.session_data["tools_cache"]
        except (ValueError, TypeError):
            return None

    def clear_tools_cache(self):
        """Clear tools cache."""
        if "tools_cache" in self.session_data:
            del self.session_data["tools_cache"]
        if "tools_cached_at" in self.session_data:
            del self.session_data["tools_cached_at"]
        self.save()

    # Protocol version

    def set_protocol_version(self, version: str):
        """Set MCP protocol version."""
        self.session_data["protocol_version"] = version

    def get_protocol_version(self) -> str:
        """Get MCP protocol version (defaults to 2025-03-26)."""
        return self.session_data.get("protocol_version", "2025-03-26")

    # Display methods

    def format_context_display(self) -> str:
        """
        Format current session context for display.

        Returns:
            Formatted string showing current auth and workspace context
        """
        lines = []

        # Auth info
        auth = self.get_auth()
        if auth:
            lines.append(f"Email: {auth['email']}")
            lines.append(f"Server: {auth['server_url']}")
        else:
            lines.append("Auth: Not configured")

        # Workspace context
        context = self.get_workspace_context()
        if context:
            lines.append(f"\nWorkspace Context:")
            lines.append(f"  Agency ID: {context['rtbm_agency_id']}")
            if "agency_name" in context:
                lines.append(f"  Agency Name: {context['agency_name']}")
            if "workspace_id" in context:
                lines.append(f"  Workspace ID: {context['workspace_id']}")
            if "workspace_name" in context:
                lines.append(f"  Workspace Name: {context['workspace_name']}")
            if "switched_at" in context:
                lines.append(f"  Switched At: {context['switched_at']}")
        else:
            lines.append("\nWorkspace Context: Not set")

        # Cache status
        cache = self.get_tools_cache()
        if cache:
            lines.append(f"\nTools Cache: {len(cache)} tools cached")

        # Last used
        last_used = self.session_data.get("last_used")
        if last_used:
            lines.append(f"Last Used: {last_used}")

        return "\n".join(lines)
