"""
Created by: Claude Code
Session ID: TBD
Date: 2025-11-24
Purpose: Tool registry for dynamic MCP tool discovery and help generation
"""

from typing import Dict, Any, List, Optional


class ToolRegistry:
    """
    Registry for dynamically discovered MCP tools.

    Fetches tools from MCP server via list_tools() and provides:
    - Tool lookup by name
    - Help text generation from schemas
    - Argument parsing from inputSchema
    """

    def __init__(self):
        self.tools: List[Dict[str, Any]] = []
        self._tools_by_name: Dict[str, Dict[str, Any]] = {}

    def load_tools(self, tools: List[Dict[str, Any]]):
        """
        Load tools from MCP server list_tools() response.

        Args:
            tools: List of tool definitions with name, description, inputSchema
        """
        self.tools = tools
        self._tools_by_name = {tool["name"]: tool for tool in tools}

    def get_tool(self, name: str) -> Optional[Dict[str, Any]]:
        """
        Get tool definition by name.

        Args:
            name: Tool name

        Returns:
            Tool definition or None if not found
        """
        return self._tools_by_name.get(name)

    def list_tool_names(self) -> List[str]:
        """Get list of all tool names."""
        return list(self._tools_by_name.keys())

    def search_tools(self, keyword: str) -> List[Dict[str, Any]]:
        """
        Search tools by keyword in name or description.

        Args:
            keyword: Search keyword (case-insensitive)

        Returns:
            List of matching tool definitions
        """
        keyword_lower = keyword.lower()
        results = []

        for tool in self.tools:
            name_match = keyword_lower in tool["name"].lower()
            desc_match = keyword_lower in tool.get("description", "").lower()

            if name_match or desc_match:
                results.append(tool)

        return results

    def generate_help(self, tool_name: str) -> str:
        """
        Generate help text for a tool from its inputSchema.

        Args:
            tool_name: Tool name

        Returns:
            Formatted help text or error message
        """
        tool = self.get_tool(tool_name)
        if not tool:
            return f"Tool '{tool_name}' not found. Use 'list-tools' to see available tools."

        lines = []
        lines.append(f"Tool: {tool['name']}")
        lines.append(f"Description: {tool.get('description', 'No description')}")
        lines.append("")

        # Parse inputSchema
        schema = tool.get("inputSchema", {})
        properties = schema.get("properties", {})
        required = schema.get("required", [])

        if properties:
            lines.append("Arguments:")
            for arg_name, arg_schema in properties.items():
                is_required = arg_name in required
                arg_type = arg_schema.get("type", "any")
                arg_desc = arg_schema.get("description", "")

                # Handle enum values
                enum_values = arg_schema.get("enum")
                if enum_values:
                    arg_type = f"enum[{', '.join(enum_values)}]"

                req_marker = "[REQUIRED]" if is_required else "[optional]"
                lines.append(f"  --{arg_name} <{arg_type}> {req_marker}")
                if arg_desc:
                    lines.append(f"      {arg_desc}")

                # Handle nested objects
                if arg_schema.get("type") == "object" and "properties" in arg_schema:
                    lines.append(f"      Nested properties:")
                    for nested_name, nested_schema in arg_schema["properties"].items():
                        nested_type = nested_schema.get("type", "any")
                        nested_desc = nested_schema.get("description", "")
                        lines.append(f"        .{nested_name} <{nested_type}>: {nested_desc}")

                # Handle arrays
                if arg_schema.get("type") == "array" and "items" in arg_schema:
                    items_schema = arg_schema["items"]
                    if isinstance(items_schema, dict):
                        items_type = items_schema.get("type", "any")
                        lines.append(f"      Array of <{items_type}>")
                        if items_type == "object" and "properties" in items_schema:
                            lines.append(f"      Item properties:")
                            for item_prop, item_schema in items_schema["properties"].items():
                                item_type = item_schema.get("type", "any")
                                item_desc = item_schema.get("description", "")
                                item_required = item_prop in items_schema.get("required", [])
                                req_str = " [required]" if item_required else ""
                                lines.append(f"        .{item_prop} <{item_type}>{req_str}: {item_desc}")
        else:
            lines.append("Arguments: None")

        return "\n".join(lines)

    def format_tools_table(self, tools: Optional[List[Dict[str, Any]]] = None) -> str:
        """
        Format tools as a table.

        Args:
            tools: List of tools to format (defaults to all tools)

        Returns:
            Formatted table string
        """
        if tools is None:
            tools = self.tools

        if not tools:
            return "No tools available"

        # Calculate column widths
        max_name_len = max(len(t["name"]) for t in tools)
        max_desc_len = 80  # Truncate descriptions

        # Header
        lines = []
        lines.append("=" * (max_name_len + max_desc_len + 5))
        lines.append(f"{'Tool Name':<{max_name_len}}  Description")
        lines.append("=" * (max_name_len + max_desc_len + 5))

        # Rows
        for tool in tools:
            name = tool["name"]
            desc = tool.get("description", "")

            # Truncate description if too long
            if len(desc) > max_desc_len:
                desc = desc[:max_desc_len - 3] + "..."

            lines.append(f"{name:<{max_name_len}}  {desc}")

        lines.append("=" * (max_name_len + max_desc_len + 5))
        lines.append(f"Total: {len(tools)} tools")

        return "\n".join(lines)

    def extract_required_args(self, tool_name: str) -> List[str]:
        """
        Extract required argument names from tool schema.

        Args:
            tool_name: Tool name

        Returns:
            List of required argument names
        """
        tool = self.get_tool(tool_name)
        if not tool:
            return []

        schema = tool.get("inputSchema", {})
        return schema.get("required", [])

    def validate_arguments(self, tool_name: str, arguments: Dict[str, Any]) -> tuple[bool, str]:
        """
        Validate arguments against tool schema.

        Args:
            tool_name: Tool name
            arguments: Arguments dict to validate

        Returns:
            (is_valid, error_message) tuple
        """
        tool = self.get_tool(tool_name)
        if not tool:
            return False, f"Tool '{tool_name}' not found"

        schema = tool.get("inputSchema", {})
        required = schema.get("required", [])
        properties = schema.get("properties", {})

        # Check required arguments
        missing = [arg for arg in required if arg not in arguments]
        if missing:
            return False, f"Missing required arguments: {', '.join(missing)}"

        # Check for unknown arguments
        unknown = [arg for arg in arguments if arg not in properties]
        if unknown:
            return False, f"Unknown arguments: {', '.join(unknown)}"

        return True, ""

    def get_tool_categories(self) -> Dict[str, List[str]]:
        """
        Categorize tools by common prefixes or patterns.

        Returns:
            Dict mapping category names to tool name lists
        """
        categories: Dict[str, List[str]] = {
            "clickhouse": [],
            "discovery": [],
            "workspace": [],
            "notion": [],
            "jira": [],
            "report": [],
            "documentation": [],
            "other": []
        }

        for tool in self.tools:
            name = tool["name"]
            categorized = False

            for category in categories:
                if category.lower() in name.lower():
                    categories[category].append(name)
                    categorized = True
                    break

            if not categorized:
                categories["other"].append(name)

        # Remove empty categories
        return {k: v for k, v in categories.items() if v}

    def format_tools_by_category(self) -> str:
        """
        Format tools grouped by category.

        Returns:
            Formatted string with categorized tools
        """
        categories = self.get_tool_categories()

        lines = []
        lines.append("=" * 80)
        lines.append("MCP Tools by Category")
        lines.append("=" * 80)

        for category, tool_names in sorted(categories.items()):
            lines.append(f"\n{category.upper()} ({len(tool_names)} tools):")
            for name in sorted(tool_names):
                tool = self.get_tool(name)
                desc = tool.get("description", "")[:60] if tool else ""
                lines.append(f"  • {name}")
                if desc:
                    lines.append(f"    {desc}...")

        lines.append("\n" + "=" * 80)
        lines.append(f"Total: {len(self.tools)} tools")

        return "\n".join(lines)
