# Limitless API Capabilities - Complete Analysis

## Thesis

This document provides comprehensive analysis of Limitless API capabilities, comparing two distinct endpoints (/v1/chats vs /v1/lifelogs), their data structures, use cases, and integration strategies including MCP servers for enhanced AI assistant access.

## Overview

Limitless offers two separate API endpoints serving different purposes: (1.0) `/v1/chats` for AI assistant conversations, and (2.0) `/v1/lifelogs` for Pendant wearable recordings with rich transcripts and audio. Section 3.0 covers MCP integration for seamless Claude access, 4.0 analyzes current implementation gaps, and 5.0 provides migration recommendations.

```mermaid
graph TD
    LimitlessAPI[Limitless API] --> Chats[/v1/chats]
    LimitlessAPI --> Lifelogs[/v1/lifelogs]

    Chats --> ChatData[AI Conversations]
    Chats --> Messages[User/Assistant Messages]
    Chats --> Summary[Chat Summaries]

    Lifelogs --> Recordings[Pendant Recordings]
    Lifelogs --> Transcripts[Speaker-Attributed Transcripts]
    Lifelogs --> Audio[Audio Files Ogg Opus]
    Lifelogs --> Markdown[Structured Markdown]

    Lifelogs --> MCP[MCP Servers]
    MCP --> Claude[Claude Desktop]
    MCP --> Windsurf[Windsurf IDE]
```

```mermaid
graph LR
    Request[API Request] --> Endpoint{Endpoint Type}
    Endpoint -->|/v1/chats| ChatResponse[Chat Messages]
    Endpoint -->|/v1/lifelogs| LifelogResponse[Lifelog Entries]
    LifelogResponse --> Process[Process Content]
    Process --> Extract[Extract Speakers]
    Extract --> Save[Save Artifacts]
```

## 1.0 Chats Endpoint (/v1/chats)

¶1 **Ordering principle:** Sections ordered by API interaction flow - endpoint details (1.1), data structure (1.2), use cases (1.3), current implementation (1.4).

---

### 1.1 Endpoint Details

¶1 **URL:** `https://api.limitless.ai/v1/chats`

¶2 **Method:** GET

¶3 **Authentication:** `X-API-Key` header with API key from .env

¶4 **Query Parameters:**
```json
{
  "date": "YYYY-MM-DD",        // Required: specific date
  "timezone": "America/Los_Angeles"  // Optional: IANA timezone, default UTC
}
```

¶5 **Rate Limiting:** 180 requests per minute per API key

---

### 1.2 Data Structure

¶1 **Response schema:**
```json
{
  "data": {
    "chats": [
      {
        "id": "string",                    // Unique chat identifier
        "summary": "string",               // AI-generated chat title
        "createdAt": "ISO8601",           // Chat creation timestamp
        "startedAt": "ISO8601",           // Optional: when chat started
        "messages": [
          {
            "id": "string",
            "text": "string",              // Message content
            "createdAt": "ISO8601",
            "user": {
              "role": "user|assistant|tool",
              "name": "string"
            },
            "toolCalls": [                 // Optional: AI tool invocations
              {
                "id": "string",
                "toolName": "string",
                "args": {}
              }
            ],
            "toolResults": [               // Optional: tool execution results
              {
                "result": {},
                "isError": false,
                "toolCallId": "string",
                "toolName": "string"
              }
            ]
          }
        ]
      }
    ]
  }
}
```

¶2 **Message roles:**
- `user` - User messages to AI assistant
- `assistant` - AI responses
- `tool` - Tool execution results

¶3 **Tool calls available:**
- `searchLifelogs` - Search lifelog entries by query and date range
- Other custom tools defined in Limitless AI assistant

---

### 1.3 Use Cases

¶1 **AI assistant conversations:**
- Daily insights generation
- Task planning queries
- Information retrieval from lifelogs
- Custom queries with AI analysis

¶2 **Conversation analysis:**
- Track AI assistant usage patterns
- Analyze question types
- Extract decisions from daily insights
- Theme detection across conversations

¶3 **Current implementation:**
- Used in our existing `00_process_conversations.py`
- Creates Knowledge Framework summaries
- Extracts individual conversations
- Theme auto-detection (Daily Insights, Longevity, M&A, etc.)

---

### 1.4 Limitations

¶1 **No audio access:** Chats endpoint does not provide audio files

¶2 **No speaker attribution:** Messages don't have detailed speaker identification

¶3 **No timestamps per word:** Only message-level timestamps

¶4 **No markdown structure:** Plain text messages without hierarchical organization

¶5 **No starred functionality:** Cannot filter or mark important chats

---

## 2.0 Lifelogs Endpoint (/v1/lifelogs)

¶1 **Ordering principle:** Sections ordered by capability hierarchy - endpoint details (2.1), rich data structure (2.2), advanced features (2.3), audio capabilities (2.4).

---

### 2.1 Endpoint Details

¶1 **URL:** `https://api.limitless.ai/v1/lifelogs`

¶2 **Method:** GET

¶3 **Authentication:** `X-API-Key` header

¶4 **Query Parameters (comprehensive):**
```json
{
  "timezone": "America/Los_Angeles",     // IANA timezone
  "date": "YYYY-MM-DD",                 // Specific date filter
  "start": "ISO8601",                   // Range start (ignores timezone param)
  "end": "ISO8601",                     // Range end (ignores timezone param)
  "cursor": "string",                   // Pagination cursor
  "direction": "asc|desc",              // Sort order (default: desc)
  "includeMarkdown": true,              // Include raw markdown (default: true)
  "includeHeadings": true,              // Include heading nodes (default: true)
  "limit": 50,                          // Max entries (default varies)
  "isStarred": false                    // Filter starred only
}
```

---

### 2.2 Rich Data Structure

¶1 **Lifelog object schema:**
```json
{
  "id": "string",
  "title": "string",                    // From first heading1 node
  "markdown": "string",                 // Full markdown content
  "contents": [                         // Hierarchical content nodes
    {
      "type": "heading1|heading2|heading3|blockquote|paragraph|etc",
      "content": "string",
      "startTime": "ISO8601",
      "endTime": "ISO8601",
      "startOffsetMs": 0,               // Millisecond offset from lifelog start
      "endOffsetMs": 1000,
      "speakerName": "Daniel",          // Optional: speaker identification
      "speakerIdentifier": "user",      // "user" or unique speaker ID
      "children": []                    // Nested content nodes
    }
  ],
  "startTime": "ISO8601",
  "endTime": "ISO8601",
  "isStarred": false,
  "updatedAt": "ISO8601"
}
```

¶2 **ContentNode types:**
- `heading1`, `heading2`, `heading3` - Hierarchical sections
- `paragraph` - Text blocks
- `blockquote` - Quoted text
- `unordered-list`, `list-item` - Lists
- `horizontal-rule` - Section dividers

¶3 **Speaker attribution:**
- Each content node can have `speakerName` and `speakerIdentifier`
- Enables multi-speaker conversation tracking
- `speakerIdentifier: "user"` marks your own speech

---

### 2.3 Advanced Features

¶1 **Hierarchical structure:**
- Content organized as tree with parent-child relationships
- Heading levels create logical sections
- Enables semantic navigation through conversations

¶2 **Precise timestamps:**
- `startOffsetMs` / `endOffsetMs` for word-level timing
- Allows audio synchronization with transcript
- Enables precise moment retrieval

¶3 **Starred lifelogs:**
- User can mark important recordings
- Filter with `isStarred: true` parameter
- Useful for quick access to key moments

¶4 **Markdown export:**
- Full markdown content preserved
- Can be used for documentation
- Maintains formatting and structure

¶5 **Pagination support:**
- `cursor` parameter for navigating large datasets
- `limit` controls entries per request
- `direction` for chronological or reverse-chronological order

---

### 2.4 Audio Capabilities

¶1 **Audio download endpoint:**
```
GET /v1/audio/download
```

¶2 **Parameters:**
```json
{
  "start": "ISO8601",          // Start time for audio segment
  "end": "ISO8601",            // End time (max 2 hours from start)
  "timezone": "America/Los_Angeles"
}
```

¶3 **Response:**
- Binary Ogg Opus audio format
- No transcoding - direct from Pendant
- Maximum 2-hour segments

¶4 **Use cases:**
- Extract specific conversation moments
- Create audio clips for sharing
- Archive important discussions
- Offline playback

---

## 3.0 MCP Integration

¶1 **Ordering principle:** Sections ordered by MCP ecosystem - protocol overview (3.1), available servers (3.2), capabilities comparison (3.3), integration guide (3.4).

---

### 3.1 Model Context Protocol Overview

¶1 **What is MCP:**
- Open standard introduced by Anthropic (November 2024)
- Standardizes AI model integration with external tools and data
- "USB-C for AI" - universal, reliable, extensible

¶2 **Benefits:**
- Unified interface for multiple AI tools (Claude, Cursor, Windsurf)
- Stateful context across conversations
- Programmatic access to lifelog data
- No need to switch between apps

---

### 3.2 Available MCP Servers

¶1 **ipvr9/mcp-limitless-server:**
- **URL:** https://github.com/ipvr9/mcp-limitless-server
- **Features:**
  - Get lifelog by ID
  - List by date
  - List by date range
  - List recent entries
  - Search recent logs (limited to 20-100 entries)
- **Target:** Claude, Windsurf, any MCP-compatible assistant

¶2 **BurtTheCoder/mcp-limitless:**
- **URL:** https://github.com/BurtTheCoder/mcp-limitless
- **Features:**
  - Hybrid semantic and keyword search
  - Full content retrieval with transcripts
  - Date-filtered listing
  - Starred item filtering
- **Security:** GitHub OAuth + Cloudflare Workers encryption
- **Target:** Claude Desktop primarily

¶3 **manueltarouca/limitless-mcp-server:**
- **URL:** https://github.com/manueltarouca/limitless-mcp-server
- **Features:**
  - Server mode
  - Client mode
  - Interactive mode for testing
- **Target:** General MCP clients

---

### 3.3 MCP Capabilities Comparison

¶1 **Advantages over direct API:**
- **Context persistence:** AI remembers previous queries
- **Semantic search:** Better than simple keyword matching
- **Integrated workflows:** Pull action items → send to Notion (via Claude)
- **No manual API calls:** Natural language queries work directly

¶2 **Limitations:**
- **Search scope:** Most servers limit to recent 20-100 entries
- **No full-history search:** Cannot scan all lifelogs efficiently
- **Rate limits apply:** Same 180 req/min as direct API
- **Requires setup:** MCP server configuration needed

---

### 3.4 MCP Integration Guide

¶1 **Setup for Claude Desktop:**

```json
{
  "mcpServers": {
    "limitless": {
      "command": "npx",
      "args": ["-y", "@ipvr9/mcp-limitless-server"],
      "env": {
        "LIMITLESS_API_KEY": "your_api_key_here"
      }
    }
  }
}
```

¶2 **Usage examples:**
```
User: "Show me my conversations about longevity from last week"
Claude: [Uses MCP limitless server to search and retrieve]

User: "Find the decision I made about Wendy yesterday"
Claude: [Searches lifelogs, extracts decision points]

User: "Get transcript from my call with Gosha on October 30"
Claude: [Retrieves specific lifelog by date and participant]
```

---

## 4.0 Current Implementation Analysis

¶1 **Ordering principle:** Sections ordered by assessment flow - what we use now (4.1), what we're missing (4.2), data type comparison (4.3).

---

### 4.1 Current Implementation

¶1 **Endpoint used:** `/v1/chats`

¶2 **Data accessed:**
- AI assistant conversations
- User messages and AI responses
- Tool calls (searchLifelogs, etc.)
- Chat summaries and timestamps

¶3 **Features implemented:**
- Date-based retrieval (PST timezone)
- Knowledge Framework summaries
- Theme auto-detection
- Individual conversation extraction
- Full transcript preservation (JSON)

¶4 **Files created:**
- `00_summary.md` - Knowledge Framework structured
- `00_full_transcript.json` - Raw API response
- `00_all_conversations_full.md` - Combined conversations
- `conversations/` - Individual chat files

---

### 4.2 Missing Capabilities

¶1 **No Pendant recordings access:**
- Cannot retrieve actual voice recordings from Limitless Pendant
- Missing real-world conversation transcripts
- No access to meeting recordings

¶2 **No audio files:**
- Cannot download audio segments
- No offline playback capability
- Missing audio archive functionality

¶3 **No speaker attribution:**
- Chat messages don't identify individual speakers
- Cannot distinguish between multiple participants
- No speaker-specific analysis

¶4 **No hierarchical structure:**
- Messages are flat, not organized by topics
- No heading-based navigation
- Missing semantic organization

¶5 **No starred functionality:**
- Cannot filter important moments
- No priority-based retrieval
- Missing quick access to key insights

¶6 **Limited search:**
- Only date-based filtering
- No semantic search within chats
- Cannot search by speaker or topic

---

### 4.3 Data Type Comparison

| Feature | /v1/chats (Current) | /v1/lifelogs (Available) |
|---------|-------------------|------------------------|
| **Data Source** | AI assistant conversations | Pendant recordings + transcripts |
| **Audio Access** | ❌ No | ✅ Yes (Ogg Opus, 2hr segments) |
| **Speaker Attribution** | ❌ No | ✅ Yes (speakerName, speakerIdentifier) |
| **Timestamps** | Message-level only | Word-level (startOffsetMs, endOffsetMs) |
| **Structure** | Flat messages | Hierarchical (heading1/2/3, nested children) |
| **Markdown** | Plain text | ✅ Full markdown with formatting |
| **Starred** | ❌ Not supported | ✅ isStarred flag |
| **Search** | Date only | Date + range + semantic (via MCP) |
| **Pagination** | ❌ Not mentioned | ✅ Cursor-based |
| **Use Case** | AI chat analysis | Real-world conversation analysis |

---

## 5.0 Migration Recommendations

¶1 **Ordering principle:** Sections ordered by implementation priority - immediate enhancements (5.1), phased migration (5.2), MCP integration (5.3), future capabilities (5.4).

---

### 5.1 Immediate Enhancements (Phase 1)

¶1 **Add lifelogs endpoint support:**

```python
def fetch_lifelogs(date_str=None, start=None, end=None, timezone="America/Los_Angeles",
                   is_starred=False, include_markdown=True, limit=50):
    """Fetch lifelogs from Limitless API"""
    api_key = os.getenv("LIMITLESS_API_KEY")
    if not api_key:
        raise ValueError("LIMITLESS_API_KEY not found in .env")

    url = "https://api.limitless.ai/v1/lifelogs"
    headers = {"X-API-Key": api_key}
    params = {
        "timezone": timezone,
        "includeMarkdown": include_markdown,
        "includeHeadings": True,
        "limit": limit
    }

    if date_str:
        params["date"] = date_str
    if start and end:
        params["start"] = start
        params["end"] = end
    if is_starred:
        params["isStarred"] = True

    response = requests.get(url, headers=headers, params=params)
    response.raise_for_status()
    return response.json()
```

¶2 **Extract speaker-attributed transcripts:**

```python
def extract_speakers_from_lifelog(lifelog):
    """Extract conversations organized by speaker"""
    speakers = defaultdict(list)

    def process_node(node):
        speaker = node.get('speakerName', 'Unknown')
        content = node.get('content', '')
        start_time = node.get('startTime')

        if content:
            speakers[speaker].append({
                'text': content,
                'time': start_time,
                'offset_ms': node.get('startOffsetMs')
            })

        # Process children recursively
        for child in node.get('children', []):
            process_node(child)

    for content_node in lifelog.get('contents', []):
        process_node(content_node)

    return dict(speakers)
```

¶3 **Create enhanced summaries with speaker analysis:**
- Add speaker distribution stats
- Track conversation flow by speaker
- Identify key speaker contributions

---

### 5.2 Phased Migration Strategy

¶1 **Phase 1 (Week 1): Dual endpoint support**
- Keep existing `/v1/chats` functionality
- Add `/v1/lifelogs` as optional mode
- Create flag: `--mode chats|lifelogs|both`
- Test with sample dates

¶2 **Phase 2 (Week 2): Audio download**
- Implement audio segment download
- Save audio files alongside transcripts
- Create playback links in markdown summaries
- Test 2-hour limit handling

¶3 **Phase 3 (Week 3): Enhanced analysis**
- Add speaker attribution to summaries
- Implement hierarchical navigation
- Create starred lifelogs quick access
- Add semantic search (local)

¶4 **Phase 4 (Week 4): MCP integration**
- Set up MCP server for Claude Desktop
- Test workflows (lifelog → Notion, etc.)
- Document MCP usage patterns
- Create skill examples for MCP queries

---

### 5.3 MCP Server Setup

¶1 **Recommended server:** `ipvr9/mcp-limitless-server`

**Why:** Most complete feature set, actively maintained, supports all major MCP clients

¶2 **Setup steps:**

**1. Install MCP server:**
```bash
npm install -g @ipvr9/mcp-limitless-server
```

**2. Configure Claude Desktop:**

Edit `~/Library/Application Support/Claude/claude_desktop_config.json`:
```json
{
  "mcpServers": {
    "limitless": {
      "command": "npx",
      "args": ["-y", "@ipvr9/mcp-limitless-server"],
      "env": {
        "LIMITLESS_API_KEY": "your_key_from_env"
      }
    }
  }
}
```

**3. Restart Claude Desktop**

**4. Test with query:**
```
User: "Show me my lifelogs from yesterday"
Claude: [Uses MCP tool automatically]
```

¶3 **Alternative: BurtTheCoder/mcp-limitless for semantic search**

If semantic search is priority, use this server:
```json
{
  "mcpServers": {
    "limitless-semantic": {
      "command": "node",
      "args": ["/path/to/mcp-limitless/dist/index.js"],
      "env": {
        "LIMITLESS_API_KEY": "your_key"
      }
    }
  }
}
```

---

### 5.4 Future Capabilities

¶1 **Audio analysis:**
- Transcribe audio segments with Whisper for verification
- Extract sentiment from voice tone
- Identify background sounds/context
- Create audio clips for key moments

¶2 **Advanced speaker analysis:**
- Speaker diarization improvements
- Speaking time distribution
- Interruption patterns
- Conversation dynamics

¶3 **Cross-reference with chats:**
- Link lifelog recordings to AI chat queries
- Track which lifelogs were referenced in chats
- Create unified timeline view

¶4 **Integration workflows:**
- Auto-create Notion tasks from action items in lifelogs
- Send daily summaries to Slack
- Archive important meetings to Google Drive
- Create searchable knowledge base

---

## 6.0 Updated Skill Architecture

¶1 **Ordering principle:** Sections ordered by implementation layers - CLI wrapper (6.1), API client (6.2), data processing (6.3), output formats (6.4).

---

### 6.1 Enhanced CLI Wrapper

¶1 **New command structure:**

```bash
# Fetch chats (existing functionality)
python 02_fetch_limitless_wrapper.py --mode chats --date 2025-11-04

# Fetch lifelogs (NEW)
python 02_fetch_limitless_wrapper.py --mode lifelogs --date 2025-11-04

# Fetch both (NEW)
python 02_fetch_limitless_wrapper.py --mode both --date 2025-11-04

# Fetch starred lifelogs only (NEW)
python 02_fetch_limitless_wrapper.py --mode lifelogs --starred

# Download audio for date (NEW)
python 02_fetch_limitless_wrapper.py --mode lifelogs --date 2025-11-04 --download-audio

# Search lifelogs (NEW)
python 02_fetch_limitless_wrapper.py --mode lifelogs --search "Gosha conversation" --range 7
```

¶2 **New flags:**
- `--mode` - Choose data source: chats | lifelogs | both
- `--starred` - Filter starred lifelogs only
- `--download-audio` - Download audio segments
- `--search` - Search query for lifelogs
- `--speakers` - Filter by speaker names

---

### 6.2 API Client Enhancements

¶1 **Unified client class:**

```python
class LimitlessClient:
    def __init__(self, api_key):
        self.api_key = api_key
        self.base_url = "https://api.limitless.ai/v1"
        self.headers = {"X-API-Key": api_key}

    def fetch_chats(self, date, timezone="America/Los_Angeles"):
        """Fetch AI assistant chats"""
        # Existing implementation
        pass

    def fetch_lifelogs(self, date=None, start=None, end=None,
                      timezone="America/Los_Angeles", is_starred=False):
        """Fetch Pendant recordings and transcripts"""
        # New implementation
        pass

    def download_audio(self, start, end, timezone="America/Los_Angeles"):
        """Download audio segment (max 2 hours)"""
        # New implementation
        pass

    def search_lifelogs(self, query, start_date, end_date):
        """Search lifelogs by content"""
        # New implementation using local search
        pass
```

---

### 6.3 Enhanced Data Processing

¶1 **Speaker extraction pipeline:**

```python
def create_lifelog_summary(lifelogs, date_str):
    """Create enhanced summary with speaker analysis"""

    # Extract all speakers
    all_speakers = set()
    speaker_content = defaultdict(list)

    for lifelog in lifelogs:
        speakers = extract_speakers_from_lifelog(lifelog)
        all_speakers.update(speakers.keys())

        for speaker, content_list in speakers.items():
            speaker_content[speaker].extend(content_list)

    # Build summary
    summary = f"""# Lifelogs Summary - {date_str}

## Thesis
Analysis of {len(lifelogs)} Pendant recordings covering {len(all_speakers)} speakers...

## 1.0 Speaker Distribution

"""

    for speaker in sorted(all_speakers):
        content = speaker_content[speaker]
        total_time = sum(c.get('duration', 0) for c in content)
        summary += f"- **{speaker}**: {len(content)} segments, {total_time}s total\n"

    return summary
```

---

### 6.4 Output Format Enhancements

¶1 **New directory structure:**

```
algorithms/A8_G&A_div/Daniel Personal/Daniel_communications/calls/{date}/
├── chats/                              # AI assistant chats (existing)
│   ├── 00_summary.md
│   ├── 00_full_transcript.json
│   └── conversations/
├── lifelogs/                           # NEW: Pendant recordings
│   ├── 00_summary.md                   # Speaker-aware summary
│   ├── 00_full_lifelogs.json          # Raw lifelog data
│   ├── recordings/                     # Individual lifelog files
│   │   ├── 01_Morning_meeting_abc.md
│   │   └── 02_Gosha_conversation_def.md
│   └── audio/                          # NEW: Audio files
│       ├── 01_Morning_meeting_abc.opus
│       └── 02_Gosha_conversation_def.opus
└── combined/                           # NEW: Unified view
    └── 00_daily_overview.md            # Cross-reference chats + lifelogs
```

---

## 7.0 Statistics

¶1 **API comparison:**
- Chats: ~47 conversations per day (Oct 30 test)
- Lifelogs: Unknown (not yet implemented)
- Rate limit: 180 req/min for both endpoints

¶2 **Data volume estimates:**
- Chats: ~2-5 MB per day (JSON)
- Lifelogs: ~10-50 MB per day (with markdown)
- Audio: ~100-500 MB per day (Ogg Opus, depends on recording time)

---

## Author Checklist

- [x] Thesis states specific outcome
- [x] Overview introduces all MECE sections
- [x] Mermaid diagrams for data flow (TD) and processing (LR)
- [x] All sections numbered (1.0 through 7.0)
- [x] Ordering principle stated for major sections
- [x] Paragraphs numbered (¶1, ¶2, etc.)
- [x] Code examples for all APIs
- [x] Comparison table for chats vs lifelogs
- [x] Migration recommendations with phases
- [x] MCP integration guide included
- [x] Cross-references to implementation files
