#!/bin/bash
# Test script for Limitless Conversations skill

set -e

SCRIPT_DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )"
PROJECT_ROOT="$( cd "$SCRIPT_DIR/../../.." && pwd )"

# Colors
GREEN='\033[0;32m'
BLUE='\033[0;34m'
RED='\033[0;31m'
NC='\033[0m' # No Color

echo -e "${BLUE}=== Testing Limitless Conversations Skill ===${NC}\n"

# Test 1: Check API key exists
echo -e "${BLUE}Test 1: Checking LIMITLESS_API_KEY in .env${NC}"
if grep -q "LIMITLESS_API_KEY" "$PROJECT_ROOT/.env"; then
    echo -e "${GREEN}✓ API key found in .env${NC}"
else
    echo -e "${RED}✗ LIMITLESS_API_KEY not found in .env${NC}"
    exit 1
fi

# Test 2: Check code files exist
echo -e "\n${BLUE}Test 2: Checking code files${NC}"
if [ -f "$PROJECT_ROOT/algorithms/A8_G&A_div/Daniel Personal/Daniel_communications/code/00_process_conversations.py" ]; then
    echo -e "${GREEN}✓ 00_process_conversations.py exists${NC}"
else
    echo -e "${RED}✗ 00_process_conversations.py not found${NC}"
    exit 1
fi

if [ -f "$PROJECT_ROOT/algorithms/A8_G&A_div/Daniel Personal/Daniel_communications/code/01_extract_conversations.py" ]; then
    echo -e "${GREEN}✓ 01_extract_conversations.py exists${NC}"
else
    echo -e "${RED}✗ 01_extract_conversations.py not found${NC}"
    exit 1
fi

if [ -f "$PROJECT_ROOT/algorithms/A8_G&A_div/Daniel Personal/Daniel_communications/code/02_fetch_limitless_wrapper.py" ]; then
    echo -e "${GREEN}✓ 02_fetch_limitless_wrapper.py exists${NC}"
else
    echo -e "${RED}✗ 02_fetch_limitless_wrapper.py not found${NC}"
    exit 1
fi

# Test 3: Check skill documentation
echo -e "\n${BLUE}Test 3: Checking skill documentation${NC}"
if [ -f "$SCRIPT_DIR/README.md" ]; then
    echo -e "${GREEN}✓ README.md exists${NC}"
else
    echo -e "${RED}✗ README.md not found${NC}"
    exit 1
fi

if [ -f "$SCRIPT_DIR/SKILL.md" ]; then
    echo -e "${GREEN}✓ SKILL.md exists${NC}"
else
    echo -e "${RED}✗ SKILL.md not found${NC}"
    exit 1
fi

# Test 4: Check Python imports
echo -e "\n${BLUE}Test 4: Testing Python imports${NC}"
cd "$PROJECT_ROOT"
python3 -c "
import sys
sys.path.append('$PROJECT_ROOT')
from algorithms.A8_G_A_div.Daniel_Personal.Daniel_communications.code.process_conversations_00 import fetch_conversations
from algorithms.A8_G_A_div.Daniel_Personal.Daniel_communications.code.extract_conversations_01 import extract_full_conversation
print('✓ Python imports successful')
" && echo -e "${GREEN}✓ Python modules import correctly${NC}" || echo -e "${RED}✗ Python import failed${NC}"

# Test 5: Show wrapper help
echo -e "\n${BLUE}Test 5: Testing wrapper script help${NC}"
python3 "$PROJECT_ROOT/algorithms/A8_G&A_div/Daniel Personal/Daniel_communications/code/02_fetch_limitless_wrapper.py" --help > /dev/null 2>&1
if [ $? -eq 0 ]; then
    echo -e "${GREEN}✓ Wrapper script help works${NC}"
else
    echo -e "${RED}✗ Wrapper script help failed${NC}"
    exit 1
fi

# Test 6: Dry run - show what would be fetched
echo -e "\n${BLUE}Test 6: Dry run - checking date parsing${NC}"
python3 -c "
from algorithms.A8_G_A_div.Daniel_Personal.Daniel_communications.code.fetch_limitless_wrapper_02 import parse_date_input
from datetime import datetime

# Test various inputs
tests = ['today', 'yesterday', '2025-11-04']
for test in tests:
    try:
        result = parse_date_input(test)
        print(f'✓ \"{test}\" → {result}')
    except Exception as e:
        print(f'✗ \"{test}\" failed: {e}')
" 2>/dev/null && echo -e "${GREEN}✓ Date parsing works${NC}" || echo -e "${RED}✗ Date parsing failed (non-critical)${NC}"

echo -e "\n${GREEN}=== All Tests Passed ===${NC}"
echo -e "\n${BLUE}Usage Examples:${NC}"
echo "  # Fetch today's conversations:"
echo "  python3 algorithms/A8_G\\&A_div/Daniel\\ Personal/Daniel_communications/code/02_fetch_limitless_wrapper.py"
echo ""
echo "  # Fetch specific date:"
echo "  python3 algorithms/A8_G\\&A_div/Daniel\\ Personal/Daniel_communications/code/02_fetch_limitless_wrapper.py 2025-11-04"
echo ""
echo "  # Fetch last 7 days:"
echo "  python3 algorithms/A8_G\\&A_div/Daniel\\ Personal/Daniel_communications/code/02_fetch_limitless_wrapper.py --range 7"
echo ""
echo "  # Ask Claude Code:"
echo "  'Get my Limitless conversations from yesterday'"
echo "  'Show me all Limitless chats from last week'"
