"""
MMM Orchestrator - Main entry point for skill

Created by: Claude Code
Session ID: Ralph Loop Execution
Date: 2025-11-26
Purpose: Coordinate all MMM skill components
"""

import argparse
import json
from pathlib import Path
from datetime import datetime

from data_analyzer import analyze_dataset, print_profile
from config_generator import generate_config_from_profile
from iteration_engine import run_iteration_engine


def run_mmm_skill(
    csv_path: str,
    output_dir: str = None,
    max_iterations: int = 10,
    conda_env: str = 'pymc_gpu_015'
) -> dict:
    """
    Main entry point for MMM Model Builder Skill.

    Args:
        csv_path: Path to MMM source CSV
        output_dir: Output directory (auto-generated if None)
        max_iterations: Maximum training iterations
        conda_env: Conda environment name

    Returns:
        Dictionary with results and artifact paths
    """
    # Auto-generate output dir if not provided
    if output_dir is None:
        timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
        output_dir = f'mmm_skill_run_{timestamp}'

    output_dir = Path(output_dir)
    output_dir.mkdir(parents=True, exist_ok=True)

    print(f"""
╔═══════════════════════════════════════════════════════════════╗
║              MMM MODEL BUILDER SKILL                          ║
║              Pure PyMC Implementation                         ║
╠═══════════════════════════════════════════════════════════════╣
║  Input:  {csv_path[:50]}...
║  Output: {output_dir}
║  Max iterations: {max_iterations}
║  Conda env: {conda_env}
╚═══════════════════════════════════════════════════════════════╝
    """)

    # Step 1: Analyze data
    print("\n[STEP 1/4] Analyzing dataset...")
    profile = analyze_dataset(csv_path)
    print_profile(profile)

    # Save profile
    profile.to_json(output_dir / 'data_profile.json')

    # Step 2: Generate initial config
    print("\n[STEP 2/4] Generating initial config...")
    config = generate_config_from_profile(profile)
    config.to_yaml(output_dir / 'initial_config.yaml')
    print(f"  Config saved to: {output_dir / 'initial_config.yaml'}")

    # Step 3: Run iteration engine
    print("\n[STEP 3/4] Running iteration engine...")
    result = run_iteration_engine(
        csv_path=csv_path,
        output_dir=str(output_dir),
        max_iterations=max_iterations,
        conda_env=conda_env
    )

    # Step 4: Generate final report
    print("\n[STEP 4/4] Generating final report...")
    report = {
        'timestamp': datetime.now().isoformat(),
        'csv_path': csv_path,
        'output_dir': str(output_dir),
        'data_profile': {
            'n_days': profile.n_days,
            'n_channels': len(profile.channels),
            'n_controls': len(profile.controls),
            'date_range': f"{profile.date_min} - {profile.date_max}",
        },
        'iterations': result['iterations'],
        'status': result['status'],
        'best_metrics': result.get('best_result', {}),
    }

    report_path = output_dir / 'skill_report.json'
    with open(report_path, 'w') as f:
        json.dump(report, f, indent=2)

    print(f"""
╔═══════════════════════════════════════════════════════════════╗
║                    SKILL EXECUTION COMPLETE                    ║
╠═══════════════════════════════════════════════════════════════╣
║  Status: {result['status'].upper():<50} ║
║  Iterations: {result['iterations']:<47} ║
║  Best R²: {result.get('best_result', {}).get('r2', 'N/A'):<49} ║
║  Report: {str(report_path)[:50]:<50} ║
╚═══════════════════════════════════════════════════════════════╝
    """)

    return report


def main():
    parser = argparse.ArgumentParser(description='MMM Model Builder Skill')
    parser.add_argument('csv_path', help='Path to MMM source CSV')
    parser.add_argument('-o', '--output', help='Output directory')
    parser.add_argument('-i', '--iterations', type=int, default=10, help='Max iterations')
    parser.add_argument('-e', '--env', default='pymc_gpu_015', help='Conda environment')

    args = parser.parse_args()

    result = run_mmm_skill(
        csv_path=args.csv_path,
        output_dir=args.output,
        max_iterations=args.iterations,
        conda_env=args.env
    )

    return result


if __name__ == '__main__':
    main()
