#!/usr/bin/env python3
"""Quick zero inbox query - optimized for speed"""
import os
import sys
from datetime import datetime, timezone

# Add project root to path
sys.path.insert(0, '$PROJECT_ROOT')

from data_sources.notion.notion_client import NotionClient

def quick_zero_inbox():
    """Fast zero inbox query with minimal output"""
    client = NotionClient()

    # Hardcoded user ID for speed (daniel@improvado.io)
    user_id = 'NOTION_USER_ID'
    db_id = '42bffb6bf5354b828750be69024d374e'

    # AI Agent Epic task ID to exclude
    ai_agent_epic_id = '2289aec6-2125-80e5-a1e8-f6b90b81fe1b'

    # Zero inbox filter - only empty Focus date AND NOT child of AI Agent Epic
    filter_query = {
        "and": [
            {
                "property": "Assignee",
                "people": {"contains": user_id}
            },
            {
                "property": "Focus date",
                "date": {"is_empty": True}
            },
            {
                "property": "Parent-task",
                "relation": {"does_not_contain": ai_agent_epic_id}
            },
            {
                "or": [
                    {"property": "Status", "status": {"equals": "In backlog"}},
                    {"property": "Status", "status": {"equals": "In progress"}}
                ]
            }
        ]
    }

    # Query with minimal output
    response = client.query_database(
        database_id=db_id,
        filter_conditions=filter_query,
        sorts=[
            {"property": "Priority", "direction": "ascending"},
            {"property": "Created time", "direction": "ascending"}
        ]
    )

    tasks = response.get('results', [])

    # Store task details for later actions
    task_list = []

    print(f"\n📥 ZERO INBOX ({len(tasks)} tasks)\n")

    for idx, task in enumerate(tasks, 1):
        props = task['properties']

        # Extract title - NO TRUNCATION
        title = 'Untitled'
        for prop_data in props.values():
            if prop_data.get('type') == 'title':
                title_texts = prop_data.get('title', [])
                if title_texts:
                    title = ''.join([t.get('plain_text', '') for t in title_texts])
                break

        # Extract priority
        priority_obj = props.get('Priority', {}).get('select', {})
        priority = priority_obj.get('name', '—') if priority_obj else '—'

        # Priority emoji
        priority_emoji = {
            'Critical': '🔴',
            'High': '🟠',
            'Medium': '🟡',
            'Low': '🟢'
        }.get(priority, '⚪')

        # Extract status
        status_obj = props.get('Status', {}).get('status', {})
        status = status_obj.get('name', '') if status_obj else ''

        # Calculate age
        created = props.get('Created time', {}).get('created_time', '')
        if created:
            created_date = datetime.fromisoformat(created.replace('Z', '+00:00'))
            age_days = (datetime.now(timezone.utc) - created_date).days
            age = f"{age_days}d"
        else:
            age = "?"

        # Store task info
        task_info = {
            'idx': idx,
            'id': task['id'],
            'title': title,
            'priority': priority,
            'status': status,
            'age': age,
            'url': task['url']
        }
        task_list.append(task_info)

        # Print task
        print(f"{idx}. {priority_emoji} [{priority}] {title}")
        print(f"   Age: {age} | Status: {status}")
        print(f"   {task['url']}")
        print()

    print(f"Total: {len(tasks)} tasks")

if __name__ == '__main__':
    quick_zero_inbox()
