---
name: notion-tasks-analyzer
description: Use when user says "show my zero inbox", "what critical tasks", or "analyze my tasks". Automatically queries Notion for unscheduled tasks, critical/urgent work, priority breakdown, and backlog health metrics. Perfect for daily planning.
---

# Notion Tasks Analyzer Skill

This skill helps you analyze and understand your Notion task backlog through different lenses: zero inbox status, critical/urgent work, and overall task health metrics.

## When to Use This Skill

Use this skill when:
- Check tasks without Focus dates (need scheduling)
- Find urgent work (Critical priority, due/overdue/unscheduled)
- Get priority breakdown, age distribution, backlog health
- Identify what needs attention today

## Quick Start Checklist

When user wants task analysis:

```markdown
[ ] 1. Determine analysis type: zero inbox, critical tasks, or full analysis
[ ] 2. Build Notion query filter (Focus date empty OR Priority Critical)
[ ] 3. Query tasks database with proper Assignee filter
[ ] 4. Calculate metrics: count by priority, age distribution
[ ] 5. Format output with clear breakdown and action items
[ ] 6. Include direct Notion URLs for each task
```

**5-Second Decision Tree:**
- User asks "zero inbox"? → Query tasks with Focus date = empty
- User asks "critical tasks"? → Query Priority = Critical AND Focus <= today
- User asks "what to work on"? → Show both zero inbox + critical

## Practical Workflow

**CRITICAL: FAST OUTPUT - NO VERBOSE LOGGING**

When user asks for "zero inbox" or "critical tasks":
1. **Use dedicated fast scripts** - `quick_zero_inbox.py` and `quick_critical_tasks.py`
2. **Clean format** - simple list with task name and URL
3. **NO API call details** - suppress all "Отправляем запрос...", "Применяются фильтры..." messages
4. **Direct execution** - run Python script directly, get clean output only

**Implementation:**

For zero inbox:
```bash
$PROJECT_ROOT/claude_venv/bin/python \
  $PROJECT_ROOT/.claude/skills/notion-tasks-analyzer/quick_zero_inbox.py
```

**Output format:**
- Clean table with: #, Task (truncated to 60 chars), Priority, Status
- Separate clickable links section with FULL task names + URLs on separate lines
- Excludes subtasks of "Daniel task to AI agent Epic" (normalized ID comparison)
- Automatically extracts titles from Name OR Summary (title type) fields

For critical tasks:
```bash
$PROJECT_ROOT/claude_venv/bin/python \
  $PROJECT_ROOT/.claude/skills/notion-tasks-analyzer/quick_critical_tasks.py
```

These scripts suppress all NotionClient verbose logging by redirecting stdout during API calls.

**Example rapid application:**
```
User: "zero inbox"

Agent output:
📥 Zero Inbox: 6 tasks (excluded 8 subtasks of 'Daniel task to AI agent Epic')
+-----+--------------------------------------------------------------+------------+------------+
|   # | Task                                                         | Priority   | Status     |
+=====+==============================================================+============+============+
|   1 | Отправить basis, prompt injection                            | None       | In backlog |
+-----+--------------------------------------------------------------+------------+------------+
|   2 | Introducing the File Search Tool in Gemini API               | None       | In backlog |
+-----+--------------------------------------------------------------+------------+------------+
|   3 | 🎯 Реализовать 17-минутную тренировку панорамного зрения для | None       | In backlog |
+-----+--------------------------------------------------------------+------------+------------+

🔗 Click to open:
  1. Отправить basis, prompt injection
     https://www.notion.so/basis-prompt-injection-2a79aec6212580df8ba7c164f2c28d5d

  2. Introducing the File Search Tool in Gemini API
     https://www.notion.so/Introducing-the-File-Search-Tool-in-Gemini-API-...

  3. 🎯 Реализовать 17-минутную тренировку панорамного зрения для снижения морганий внимания
     https://www.notion.so/17-2a89aec62125815cb2adddc860c00be2
```

## Analysis Types

### 1. Zero Inbox Analysis

**What is "Zero Inbox"?**
All your active tasks have a Focus date assigned. Tasks in your inbox need attention.

**Zero Inbox Criteria:**
- ✅ Assigned to YOU (from NOTION_YOUR_EMAIL)
- ✅ Focus date is **EMPTY** (null/not set) ONLY
- ✅ Status is **active** (In backlog OR In progress)
- ✅ **EXCLUDES** subtasks of "Daniel task to AI agent Epic" (ID: 2289aec6212580e5a1e8f6b90b81fe1b)
- ✅ **INCLUDES** all other subtasks and top-level tasks

**Note:** Zero inbox shows ALL tasks without Focus date, including subtasks from other parent tasks. Only subtasks of the specific "Daniel task to AI agent Epic" are filtered out.

**What You'll See:**
- Total zero inbox count
- Priority breakdown (Critical, High, Medium, Low)
- Age distribution (tasks aging in backlog)
- Oldest unscheduled tasks
- Tasks due today or overdue

**Trigger phrases:**
- "Show my zero inbox"
- "What tasks need scheduling?"
- "Check my unscheduled work"
- "Tasks without focus dates"

### 2. Critical Tasks Analysis

**What are "Critical Tasks"?**
Tasks with Priority = "Critical" that need immediate attention (due today, overdue, or unscheduled).

**Critical Tasks Criteria:**
- ✅ Assigned to YOU
- ✅ Priority is **Critical**
- ✅ Status is **In backlog** OR **In progress**
- ✅ Focus date is **<= today** OR **empty**

**What You'll See:**
- All critical tasks needing attention
- Sorted by age (oldest first)
- Focus date status (today/overdue/unscheduled)
- Task creation date and age
- Direct Notion URLs for each task

**Trigger phrases:**
- "Show my critical tasks"
- "What critical work do I need to do today?"
- "List urgent items"
- "Check critical priority tasks"

### 3. Combined Analysis

Get comprehensive view of your task health:
- Zero inbox count
- Critical tasks count
- Priority distribution (all priorities)
- Age metrics (oldest, average age)
- Status breakdown
- Recommendations for scheduling

**Trigger phrases:**
- "Analyze my tasks"
- "Task health report"
- "Backlog overview"
- "What should I work on?"

## Authentication & Setup

### Required Environment Variables

```bash
# .env file
NOTION_TOKEN="secret_abc123..."           # Required: Integration token
NOTION_YOUR_EMAIL="your.email@domain.com" # Required: For task filtering
```

### Getting Your Notion Token

1. Go to https://www.notion.so/my-integrations
2. Create new integration or use existing
3. Copy the "Internal Integration Token"
4. Set as `NOTION_TOKEN` environment variable

### Important Constants

- **Tasks Database ID:** `42bffb6bf5354b828750be69024d374e`
- **Your User ID:** Automatically looked up via NOTION_YOUR_EMAIL

## Implementation Details

### Python Client Usage

```python
from data_sources.notion.notion_client import NotionClient

client = NotionClient()

# Zero Inbox Query - ONLY tasks without Focus date
zero_inbox_filter = {
    "and": [
        {
            "property": "Assignee",
            "people": {"contains": YOUR_USER_ID}
        },
        {
            "property": "Focus date",
            "date": {"is_empty": True}
        },
        {
            "or": [
                {"property": "Status", "status": {"equals": "In backlog"}},
                {"property": "Status", "status": {"equals": "In progress"}}
            ]
        }
    ]
}

results = client.query_database(
    database_id="42bffb6bf5354b828750be69024d374e",
    filter_conditions=zero_inbox_filter,
    sorts=[
        {"property": "Priority", "direction": "ascending"},
        {"property": "Created time", "direction": "ascending"}
    ]
)

# Critical Tasks Query
critical_filter = {
    "and": [
        {
            "property": "Assignee",
            "people": {"contains": YOUR_USER_ID}
        },
        {
            "property": "Priority",
            "select": {"equals": "Critical"}
        },
        {
            "or": [
                {"property": "Focus date", "date": {"is_empty": True}},
                {"property": "Focus date", "date": {"on_or_before": "today"}}
            ]
        },
        {
            "or": [
                {"property": "Status", "status": {"equals": "In backlog"}},
                {"property": "Status", "status": {"equals": "In progress"}}
            ]
        }
    ]
}

critical_tasks = client.query_database(
    database_id="42bffb6bf5354b828750be69024d374e",
    filter_conditions=critical_filter,
    sorts=[
        {"property": "Created time", "direction": "ascending"}
    ]
)
```

### CLI Usage

```bash
# Zero Inbox
python -c "
from data_sources.notion.notion_client import NotionClient
client = NotionClient()
# ... (use filter from above)
"

# Critical Tasks
python -c "
from data_sources.notion.notion_client import NotionClient
client = NotionClient()
# ... (use critical filter)
"
```

## Output Format

### Zero Inbox Report

```
📥 ZERO INBOX - Tasks Needing Attention
========================================

📊 Summary:
   Total: 15 tasks without Focus date
   Critical: 2, High: 5, Medium: 6, Low: 2

🔴 CRITICAL (2 tasks):
   1. Fix production bug - Age: 3 days
      URL: https://notion.so/...

🟠 HIGH (5 tasks):
   1. Complete Q4 planning - Age: 7 days
      URL: https://notion.so/...
```

### Critical Tasks Report

```
🔴 CRITICAL TASKS - Immediate Action Required
============================================

Total: 3 tasks needing attention

1. [OVERDUE] Fix production outage - 5 days old
   Focus date: 2 days ago
   URL: https://notion.so/...

2. [TODAY] Complete security audit - 10 days old
   Focus date: Today
   URL: https://notion.so/...

3. [UNSCHEDULED] Review access controls - 15 days old
   Focus date: Not set
   URL: https://notion.so/...
```

## Troubleshooting

### "No tasks found"
- ✅ Check NOTION_YOUR_EMAIL matches your Notion account
- ✅ Verify integration has access to Tasks database
- ✅ Ensure you have tasks assigned to you

### "Authentication failed"
- ✅ Verify NOTION_TOKEN is valid (starts with "secret_")
- ✅ Check integration hasn't been revoked
- ✅ Ensure token has correct permissions

### "Database query failed"
- ✅ Verify database ID: `42bffb6bf5354b828750be69024d374e`
- ✅ Check integration has access to database
- ✅ Ensure required properties exist (Focus date, Priority, Status, Assignee)

## Best Practices

1. **Daily Routine**:
   - Morning: Check critical tasks
   - Afternoon: Review zero inbox, schedule tasks
   - Evening: Ensure tomorrow's tasks have Focus dates

2. **Priority Management**:
   - Use "Critical" sparingly (truly urgent only)
   - Most work should be High/Medium
   - Low priority for nice-to-haves

3. **Focus Date Strategy**:
   - Every task should have a Focus date
   - Don't over-schedule (be realistic)
   - Review and adjust weekly

4. **Task Aging**:
   - Tasks >30 days old need review (still relevant?)
   - Tasks >90 days old should be archived or deleted
   - Keep backlog fresh and actionable

## Related Skills

- **notion-tasks-operations**: Create, update, query tasks (CRUD operations)
- **claude-code-sessions**: Search past Claude Code sessions
- **fireflies-calls**: Analyze call transcripts and recordings

## Examples

### Example 1: Morning Planning

```
User: "What do I need to work on today?"

Output:
🔴 CRITICAL TASKS: 2 tasks
🟠 HIGH PRIORITY: 5 tasks
📥 ZERO INBOX: 12 tasks need scheduling

Recommended action:
1. Handle 2 critical tasks first
2. Schedule the 12 unscheduled tasks
3. Work through high priority items
```

### Example 2: Zero Inbox Check

```
User: "Show my zero inbox"

Output:
📥 Zero Inbox Status: 15 tasks

Priority Breakdown:
- Critical: 2 tasks (oldest: 5 days)
- High: 7 tasks (oldest: 12 days)
- Medium: 4 tasks (oldest: 20 days)
- Low: 2 tasks (oldest: 45 days)

Oldest Unscheduled:
1. Update documentation - 45 days old
2. Review API endpoints - 20 days old
...
```

### Example 3: Critical Tasks Only

```
User: "Show critical tasks"

Output:
🔴 3 Critical Tasks Need Attention

[OVERDUE] Production bug fix - 3 days old
          Due: 2 days ago
          URL: https://notion.so/...

[TODAY] Security review - 1 day old
        Due: Today
        URL: https://notion.so/...

[UNSCHEDULED] Incident response plan - 7 days old
              No Focus date set
              URL: https://notion.so/...
```

## Technical Notes

- Uses Notion API v2022-06-28
- Queries Tasks database: `42bffb6bf5354b828750be69024d374e`
- Filters by Assignee (your user ID)
- Sorts by Priority (ascending) then Created time (oldest first)
- Excludes completed/closed tasks
- Age calculated from Created time to now

## Version History

- v1.0 (Nov 2025): Initial unified analyzer (merged zero-inbox + critical-tasks + critical-tasks-analyzer)
