"""
Created by: Claude Code
Session ID: current
Date: 2025-01-12
Purpose: Fast critical tasks query with clean output - no verbose logging
"""

import os
import sys
import io
from datetime import datetime, timezone
from dotenv import load_dotenv

# Load environment
load_dotenv()
sys.path.insert(0, '$PROJECT_ROOT')

from data_sources.notion.notion_client import NotionClient
from data_sources.notion.agent_commands.user_commands import get_my_user_info

def get_critical_tasks_fast():
    """Get critical tasks with minimal output - suppress all NotionClient verbose logging"""

    # Redirect stdout to suppress NotionClient prints
    old_stdout = sys.stdout
    sys.stdout = io.StringIO()

    try:
        # Get user info
        user_info = get_my_user_info()

        if not user_info:
            sys.stdout = old_stdout
            print("ERROR: Could not find user info. Check NOTION_YOUR_EMAIL in .env")
            return

        # Initialize client
        client = NotionClient()
        tasks_db_id = '42bffb6bf5354b828750be69024d374e'

        # Critical tasks filter
        critical_filter = {
            'and': [
                {
                    'property': 'Assignee',
                    'people': {'contains': user_info['id']}
                },
                {
                    'property': 'Priority',
                    'select': {'equals': 'Critical'}
                },
                {
                    'or': [
                        {'property': 'Status', 'status': {'equals': 'In backlog'}},
                        {'property': 'Status', 'status': {'equals': 'In progress'}}
                    ]
                }
            ]
        }

        # Query database
        results = client.query_database(
            database_id=tasks_db_id,
            filter_conditions=critical_filter,
            sorts=[
                {'property': 'Created time', 'direction': 'ascending'}
            ]
        )

        tasks = results.get('results', [])

        # Restore stdout
        sys.stdout = old_stdout

        # Categorize by Focus date
        overdue = []
        today_tasks = []
        future = []
        unscheduled = []

        today = datetime.now(timezone.utc).date()

        for task in tasks:
            props = task.get('properties', {})
            focus_date_prop = props.get('Focus date', {})

            if not focus_date_prop.get('date'):
                unscheduled.append(task)
            else:
                focus_date_str = focus_date_prop['date'].get('start')
                if focus_date_str:
                    focus_date = datetime.fromisoformat(focus_date_str.replace('Z', '+00:00')).date()
                    if focus_date < today:
                        overdue.append(task)
                    elif focus_date == today:
                        today_tasks.append(task)
                    else:
                        future.append(task)

        # Print clean output
        print(f"\n🔴 Critical Tasks: {len(tasks)} total")
        print(f"   Overdue: {len(overdue)} | Today: {len(today_tasks)} | Unscheduled: {len(unscheduled)} | Future: {len(future)}\n")

        if len(tasks) == 0:
            print("✅ No critical tasks!\n")
            return

        def print_task(task, status_label):
            props = task.get('properties', {})

            # Get title - try multiple fields
            title = 'Untitled'

            # Try Name property first
            title_prop = props.get('Name', {})
            if title_prop.get('title') and len(title_prop['title']) > 0:
                title = title_prop['title'][0].get('plain_text', '').strip()

            # If still empty, try Summary property
            if not title or title == 'Untitled':
                summary_prop = props.get('Summary', {})
                if summary_prop.get('rich_text') and len(summary_prop['rich_text']) > 0:
                    title = summary_prop['rich_text'][0].get('plain_text', '').strip()

            # If still empty, extract from URL
            if not title or title == 'Untitled':
                url_temp = task.get('url', '')
                if url_temp:
                    # Extract readable part from URL (after last /)
                    url_parts = url_temp.rstrip('/').split('/')
                    if url_parts:
                        title = url_parts[-1].replace('-', ' ')[:80]  # Limit length

            # Get created time (age)
            created_time = props.get('Created time', {}).get('created_time')
            age_days = 0
            if created_time:
                created_dt = datetime.fromisoformat(created_time.replace('Z', '+00:00'))
                age_days = (datetime.now(timezone.utc) - created_dt).days

            # Get focus date
            focus_date_prop = props.get('Focus date', {})
            focus_date_str = 'Not set'
            if focus_date_prop.get('date'):
                focus_date_str = focus_date_prop['date'].get('start', 'Not set')

            # Get URL
            url = task.get('url', '')

            print(f"[{status_label}] {title} ({age_days}d old, focus: {focus_date_str})")
            print(f"        {url}\n")

        if overdue:
            print(f"⚠️  OVERDUE ({len(overdue)}):")
            for task in overdue[:5]:  # Limit to 5
                print_task(task, 'OVERDUE')
            if len(overdue) > 5:
                print(f"    ... and {len(overdue) - 5} more overdue\n")

        if today_tasks:
            print(f"📅 TODAY ({len(today_tasks)}):")
            for task in today_tasks:
                print_task(task, 'TODAY')

        if unscheduled:
            print(f"❓ UNSCHEDULED ({len(unscheduled)}):")
            for task in unscheduled:
                print_task(task, 'UNSCHEDULED')

        if future:
            print(f"⏭️  FUTURE ({len(future)}):")
            for task in future[:3]:  # Limit to 3
                print_task(task, 'FUTURE')
            if len(future) > 3:
                print(f"    ... and {len(future) - 3} more future\n")

    except Exception as e:
        # Restore stdout on error
        sys.stdout = old_stdout
        print(f"ERROR: {e}")
        import traceback
        traceback.print_exc()

if __name__ == '__main__':
    get_critical_tasks_fast()
