"""
Created by: Claude Code
Session ID: current
Date: 2025-01-12
Purpose: Fast zero inbox query with clean table output - no verbose logging
"""

import os
import sys
import io
from datetime import datetime, timezone
from dotenv import load_dotenv
from tabulate import tabulate

# Load environment
load_dotenv()
sys.path.insert(0, '$PROJECT_ROOT')

from data_sources.notion.notion_client import NotionClient
from data_sources.notion.agent_commands.user_commands import get_my_user_info

def get_zero_inbox_fast():
    """Get zero inbox with minimal output - suppress all NotionClient verbose logging"""

    # Redirect stdout to suppress NotionClient prints
    old_stdout = sys.stdout
    sys.stdout = io.StringIO()

    try:
        # Get user info
        user_info = get_my_user_info()

        if not user_info:
            sys.stdout = old_stdout
            print("ERROR: Could not find user info. Check NOTION_YOUR_EMAIL in .env")
            return

        # Initialize client
        client = NotionClient()
        tasks_db_id = '42bffb6bf5354b828750be69024d374e'

        # Zero inbox filter - exclude ONLY subtasks of "Daniel task to AI agent Epic"
        # This specific parent task ID should be excluded
        EXCLUDED_PARENT_TASK_ID = '2289aec6212580e5a1e8f6b90b81fe1b'

        zero_inbox_filter = {
            'and': [
                {
                    'property': 'Assignee',
                    'people': {'contains': user_info['id']}
                },
                {
                    'property': 'Focus date',
                    'date': {'is_empty': True}
                },
                {
                    'or': [
                        {'property': 'Status', 'status': {'equals': 'In backlog'}},
                        {'property': 'Status', 'status': {'equals': 'In progress'}}
                    ]
                }
            ]
        }

        # Query database (this produces verbose output we're suppressing)
        results = client.query_database(
            database_id=tasks_db_id,
            filter_conditions=zero_inbox_filter,
            sorts=[
                {'property': 'Priority', 'direction': 'ascending'},
                {'property': 'Created time', 'direction': 'ascending'}
            ]
        )

        all_tasks = results.get('results', [])

        # Filter out subtasks of the excluded parent task
        # Normalize IDs by removing dashes for comparison
        excluded_id_normalized = EXCLUDED_PARENT_TASK_ID.replace('-', '')

        tasks = []
        for task in all_tasks:
            props = task.get('properties', {})
            parent_prop = props.get('Parent-task', {})
            parent_relations = parent_prop.get('relation', [])

            # Check if this task is a subtask of the excluded parent
            is_excluded_subtask = any(
                rel.get('id', '').replace('-', '') == excluded_id_normalized
                for rel in parent_relations
            )

            if not is_excluded_subtask:
                tasks.append(task)

        # Restore stdout
        sys.stdout = old_stdout

        # Print clean output
        excluded_count = len(all_tasks) - len(tasks)
        print(f"\n📥 Zero Inbox: {len(tasks)} tasks", end='')
        if excluded_count > 0:
            print(f" (excluded {excluded_count} subtasks of 'Daniel task to AI agent Epic')")
        else:
            print()

        if len(tasks) == 0:
            print("✅ Great! All tasks are scheduled.\n")
            return

        # Sort by priority (Critical > High > Medium > Low > None)
        priority_order = {'Critical': 0, 'High': 1, 'Medium': 2, 'Low': 3, 'None': 4}

        def get_priority_order(task):
            props = task.get('properties', {})
            priority_prop = props.get('Priority', {})
            if priority_prop.get('select'):
                priority = priority_prop['select']['name']
                return priority_order.get(priority, 4)
            return 4  # None priority

        tasks_sorted = sorted(tasks, key=get_priority_order)

        # Prepare table data
        table_data = []
        for i, task in enumerate(tasks_sorted, 1):
            props = task.get('properties', {})

            # Get title - try multiple fields
            title = 'Untitled'

            # Try Name property first
            title_prop = props.get('Name', {})
            if title_prop.get('title') and len(title_prop['title']) > 0:
                title = title_prop['title'][0].get('plain_text', '').strip()

            # If still empty, try Summary property (can be title OR rich_text)
            if not title or title == 'Untitled':
                summary_prop = props.get('Summary', {})
                # Check if Summary is a title field
                if summary_prop.get('type') == 'title' and summary_prop.get('title'):
                    if len(summary_prop['title']) > 0:
                        title = summary_prop['title'][0].get('plain_text', '').strip()
                # Or if it's a rich_text field
                elif summary_prop.get('rich_text') and len(summary_prop['rich_text']) > 0:
                    title = summary_prop['rich_text'][0].get('plain_text', '').strip()

            # If still empty, extract from URL
            if not title or title == 'Untitled':
                url_temp = task.get('url', '')
                if url_temp:
                    # Extract readable part from URL (after last /)
                    url_parts = url_temp.rstrip('/').split('/')
                    if url_parts:
                        title = url_parts[-1].replace('-', ' ')[:60]  # Limit length

            # Get URL
            url = task.get('url', '')

            # Get priority
            priority_prop = props.get('Priority', {})
            priority = 'None'
            if priority_prop.get('select'):
                priority = priority_prop['select']['name']

            # Get status
            status_prop = props.get('Status', {})
            status = 'Unknown'
            if status_prop.get('status'):
                status = status_prop['status']['name']

            # Add to table WITHOUT URL (for clean display)
            table_data.append([i, title[:60], priority, status])

        # Print table WITHOUT URLs
        headers = ['#', 'Task', 'Priority', 'Status']
        print(tabulate(table_data, headers=headers, tablefmt='grid', maxcolwidths=[3, 60, 10, 15]))

        # Print clickable URLs with FULL task names
        print("\n🔗 Click to open:")
        for i, task in enumerate(tasks_sorted, 1):
            props = task.get('properties', {})

            # Get FULL title (same extraction logic but without truncation)
            title = 'Untitled'

            # Try Name property first
            title_prop = props.get('Name', {})
            if title_prop.get('title') and len(title_prop['title']) > 0:
                title = title_prop['title'][0].get('plain_text', '').strip()

            # Try Summary property
            if not title or title == 'Untitled':
                summary_prop = props.get('Summary', {})
                if summary_prop.get('type') == 'title' and summary_prop.get('title'):
                    if len(summary_prop['title']) > 0:
                        title = summary_prop['title'][0].get('plain_text', '').strip()
                elif summary_prop.get('rich_text') and len(summary_prop['rich_text']) > 0:
                    title = summary_prop['rich_text'][0].get('plain_text', '').strip()

            # Extract from URL as fallback
            if not title or title == 'Untitled':
                url_temp = task.get('url', '')
                if url_temp:
                    url_parts = url_temp.rstrip('/').split('/')
                    if url_parts:
                        title = url_parts[-1].replace('-', ' ')

            url = task.get('url', '')
            print(f"  {i}. {title}")
            print(f"     {url}\n")

    except Exception as e:
        # Restore stdout on error
        sys.stdout = old_stdout
        print(f"ERROR: {e}")
        import traceback
        traceback.print_exc()

if __name__ == '__main__':
    get_zero_inbox_fast()
