#!/bin/bash

# Test script for Notion Tasks Analyzer skill

set -e

echo "🧪 Testing Notion Tasks Analyzer Skill"
echo "========================================"
echo ""

# Colors
GREEN='\033[0;32m'
RED='\033[0;31m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Check environment
echo -e "${BLUE}Test 1: Checking NOTION_TOKEN in .env${NC}"
if grep -q "NOTION_TOKEN" .env 2>/dev/null; then
    echo -e "${GREEN}✓ NOTION_TOKEN found in .env${NC}"
else
    echo -e "${RED}✗ NOTION_TOKEN not found in .env${NC}"
    echo "Add NOTION_TOKEN to .env file"
    exit 1
fi

echo -e "${BLUE}Test 2: Checking NOTION_YOUR_EMAIL in .env${NC}"
if grep -q "NOTION_YOUR_EMAIL" .env 2>/dev/null; then
    echo -e "${GREEN}✓ NOTION_YOUR_EMAIL found in .env${NC}"
else
    echo -e "${RED}✗ NOTION_YOUR_EMAIL not found in .env${NC}"
    echo "Add NOTION_YOUR_EMAIL to .env file"
    exit 1
fi

# Check Python client
echo -e "\n${BLUE}Test 3: Checking Notion client${NC}"
if python3 -c "from data_sources.notion.notion_client import NotionClient; print('✓ NotionClient imported successfully')" 2>/dev/null; then
    echo -e "${GREEN}✓ Notion client available${NC}"
else
    echo -e "${RED}✗ Failed to import NotionClient${NC}"
    echo "Check data_sources/notion/notion_client.py exists"
    exit 1
fi

# Test database connection
echo -e "\n${BLUE}Test 4: Testing database connection${NC}"
python3 -c "
import sys
sys.path.insert(0, '.')
from data_sources.notion.notion_client import NotionClient

try:
    client = NotionClient()
    print('✓ Successfully connected to Notion')
except Exception as e:
    print(f'✗ Connection failed: {e}')
    sys.exit(1)
" && echo -e "${GREEN}✓ Database connection successful${NC}" || echo -e "${RED}✗ Database connection failed${NC}"

# Test zero inbox query
echo -e "\n${BLUE}Test 5: Testing zero inbox query${NC}"
python3 -c "
import sys
import os
sys.path.insert(0, '.')
from data_sources.notion.notion_client import NotionClient
from datetime import datetime

try:
    client = NotionClient()

    # Get user ID
    your_email = os.getenv('NOTION_YOUR_EMAIL', 'daniel@improvado.io')
    user_id = client.get_user_id_by_email(your_email)

    # Query zero inbox
    zero_inbox_filter = {
        'and': [
            {'property': 'Assignee', 'people': {'contains': user_id}},
            {
                'or': [
                    {'property': 'Focus date', 'date': {'is_empty': True}},
                    {'property': 'Focus date', 'date': {'on_or_before': datetime.now().strftime('%Y-%m-%d')}}
                ]
            },
            {
                'or': [
                    {'property': 'Status', 'status': {'equals': 'In backlog'}},
                    {'property': 'Status', 'status': {'equals': 'In progress'}}
                ]
            }
        ]
    }

    results = client.query_database(
        database_id='42bffb6bf5354b828750be69024d374e',
        filter_conditions=zero_inbox_filter
    )

    count = len(results.get('results', []))
    print(f'✓ Zero inbox query successful: {count} tasks found')
except Exception as e:
    print(f'✗ Zero inbox query failed: {e}')
    sys.exit(1)
" && echo -e "${GREEN}✓ Zero inbox query works${NC}" || echo -e "${RED}✗ Zero inbox query failed${NC}"

# Test critical tasks query
echo -e "\n${BLUE}Test 6: Testing critical tasks query${NC}"
python3 -c "
import sys
import os
sys.path.insert(0, '.')
from data_sources.notion.notion_client import NotionClient
from datetime import datetime

try:
    client = NotionClient()

    # Get user ID
    your_email = os.getenv('NOTION_YOUR_EMAIL', 'daniel@improvado.io')
    user_id = client.get_user_id_by_email(your_email)

    # Query critical tasks
    critical_filter = {
        'and': [
            {'property': 'Assignee', 'people': {'contains': user_id}},
            {'property': 'Priority', 'select': {'equals': 'Critical'}},
            {
                'or': [
                    {'property': 'Focus date', 'date': {'is_empty': True}},
                    {'property': 'Focus date', 'date': {'on_or_before': datetime.now().strftime('%Y-%m-%d')}}
                ]
            },
            {
                'or': [
                    {'property': 'Status', 'status': {'equals': 'In backlog'}},
                    {'property': 'Status', 'status': {'equals': 'In progress'}}
                ]
            }
        ]
    }

    results = client.query_database(
        database_id='42bffb6bf5354b828750be69024d374e',
        filter_conditions=critical_filter
    )

    count = len(results.get('results', []))
    print(f'✓ Critical tasks query successful: {count} tasks found')
except Exception as e:
    print(f'✗ Critical tasks query failed: {e}')
    sys.exit(1)
" && echo -e "${GREEN}✓ Critical tasks query works${NC}" || echo -e "${RED}✗ Critical tasks query failed${NC}"

echo ""
echo -e "${GREEN}========================================${NC}"
echo -e "${GREEN}✅ All tests passed!${NC}"
echo -e "${GREEN}========================================${NC}"
echo ""
echo "Skill is ready to use. Try:"
echo "  - 'Show my zero inbox'"
echo "  - 'Show critical tasks'"
echo "  - 'What should I work on?'"
