# Advanced Notion Patterns

This document contains advanced patterns extracted from the chrome-extension-tcs codebase.

## Pattern 1: Hierarchical Task Management

### Use Case: Epic → Feature → Task Hierarchy

From: `data_sources/notion/hierarchy_management_from_A8/`

```python
from data_sources.notion.notion_client import NotionClient

client = NotionClient()

# Check task hierarchy validity
def validate_task_hierarchy(task_id: str):
    """Ensure task has proper parent-child structure"""

    # Get full tree
    tree = client.get_task_tree_with_relations(
        page_id=task_id,
        parent_depth=5,
        child_depth=10
    )

    # Check for circular references
    visited = set()
    def check_circular(node, path=None):
        if path is None:
            path = []

        node_id = node['id']
        if node_id in path:
            raise ValueError(f"Circular reference detected: {path + [node_id]}")

        for child in node.get('children', []):
            check_circular(child, path + [node_id])

    check_circular(tree)
    return tree

# Find orphaned tasks (no parent)
def find_orphan_tasks(assignee: str):
    """Find tasks without parent relationships"""
    from data_sources.notion.notion_cli import NotionAgent

    agent = NotionAgent()
    tasks = agent.get_tasks(assignee=assignee, status="In Progress")

    orphans = []
    for task in tasks:
        properties = task.get('properties', {})
        parent_task = properties.get('Parent-task', {}).get('relation', [])

        if not parent_task:
            orphans.append(task)

    return orphans
```

## Pattern 2: Bulk Task Updates

### Use Case: Mass status updates for related tasks

```python
from data_sources.notion.notion_client import NotionClient
from data_sources.notion.notion_cli import NotionAgent

def bulk_update_subtask_status(parent_task_id: str, new_status: str):
    """Update all subtasks when parent status changes"""

    agent = NotionAgent()
    client = NotionClient()

    # Get all children
    tree = agent.get_task_tree(parent_task_id, child_depth=3)

    def collect_children(node):
        """Recursively collect all child IDs"""
        children = []
        for child in node.get('children', []):
            children.append(child['id'])
            children.extend(collect_children(child))
        return children

    child_ids = collect_children(tree)

    # Update all children
    for child_id in child_ids:
        try:
            agent.update_task_status(child_id, new_status)
            print(f"✓ Updated {child_id}")
        except Exception as e:
            print(f"✗ Failed {child_id}: {e}")
```

## Pattern 3: Task Deduplication

### Use Case: Merge duplicate tasks

From: `data_sources/notion/merge_tasks/`

```python
from data_sources.notion.notion_client import NotionClient

def find_duplicate_tasks(title_search: str):
    """Find tasks with similar titles"""
    from data_sources.notion.notion_cli import NotionAgent

    agent = NotionAgent()
    pages = agent.search_pages(title_search)

    # Group by exact title match
    duplicates = {}
    for page in pages:
        title_prop = page.get('properties', {}).get('title', {})
        title_text = title_prop.get('title', [{}])[0].get('text', {}).get('content', '')

        if title_text:
            if title_text not in duplicates:
                duplicates[title_text] = []
            duplicates[title_text].append(page['id'])

    # Return only groups with duplicates
    return {k: v for k, v in duplicates.items() if len(v) > 1}

def migrate_task_relations(from_task_id: str, to_task_id: str):
    """Move all relations from one task to another before deletion"""

    client = NotionClient()

    # Get source task
    source = client.get_page(from_task_id)
    source_props = source.get('properties', {})

    # Get all relation fields
    relations_to_migrate = {}
    for prop_name, prop_value in source_props.items():
        if prop_value.get('type') == 'relation':
            relations = prop_value.get('relation', [])
            if relations:
                relations_to_migrate[prop_name] = relations

    # Apply to target task
    if relations_to_migrate:
        client.update_page_properties(to_task_id, {
            prop_name: {'relation': relations}
            for prop_name, relations in relations_to_migrate.items()
        })

    print(f"Migrated {len(relations_to_migrate)} relation fields")
```

## Pattern 4: Rich Content Tasks

### Use Case: Tasks with markdown descriptions, code blocks, images

```python
from data_sources.notion.markdown_to_blocks import markdown_to_notion_blocks
from data_sources.notion.notion_cli import NotionAgent

def create_task_with_rich_description(title: str, markdown_description: str):
    """Create task with formatted markdown content"""

    agent = NotionAgent()

    # Create task with markdown
    task = agent.create_task(
        title=title,
        description_md=markdown_description,  # Uses markdown_to_blocks internally
        assignee="daniel_kravtsov",
        status="In backlog"
    )

    return task

# Example markdown
markdown = """
## Implementation Steps

1. Create database schema
2. Implement API endpoints
   - `GET /api/tasks`
   - `POST /api/tasks`
3. Add frontend components

### Code Example

```python
def get_tasks():
    return Task.query.all()
```

**Note:** Remember to add tests!
"""

task = create_task_with_rich_description("Implement Tasks API", markdown)
```

## Pattern 5: Task Export and Documentation

### Use Case: Export task trees to markdown for documentation

```python
from data_sources.notion.notion_cli import NotionAgent

def export_task_tree_to_markdown(root_task_id: str, output_file: str):
    """Export entire task tree as markdown document"""

    agent = NotionAgent()
    tree = agent.get_task_tree(root_task_id, child_depth=10)

    def tree_to_markdown(node, level=0):
        """Recursively convert tree to markdown"""
        indent = "  " * level
        title = node.get('title', 'Untitled')
        status = node.get('status', 'Unknown')
        assignee = node.get('assignee', 'Unassigned')

        # Format node
        md = f"{indent}- **{title}**\n"
        md += f"{indent}  - Status: {status}\n"
        md += f"{indent}  - Assignee: {assignee}\n"

        # Add children
        for child in node.get('children', []):
            md += tree_to_markdown(child, level + 1)

        return md

    markdown = f"# {tree.get('title', 'Task Tree')}\n\n"
    markdown += tree_to_markdown(tree)

    with open(output_file, 'w') as f:
        f.write(markdown)

    print(f"Exported to {output_file}")
```

## Pattern 6: Task Analytics

### Use Case: Generate statistics about task completion

```python
from data_sources.notion.notion_cli import NotionAgent
from datetime import datetime, timedelta

def analyze_task_completion(assignee: str, days: int = 30):
    """Analyze task completion metrics"""

    agent = NotionAgent()

    # Get all tasks from last N days
    tasks = agent.get_tasks(assignee=assignee, days_back=days)

    # Categorize by status
    stats = {
        'total': len(tasks),
        'done': 0,
        'in_progress': 0,
        'backlog': 0,
        'by_priority': {'High': 0, 'Medium': 0, 'Low': 0},
        'avg_completion_time': None
    }

    completion_times = []

    for task in tasks:
        props = task.get('properties', {})

        # Status
        status = props.get('Status', {}).get('status', {}).get('name', '')
        if status == 'Done':
            stats['done'] += 1

            # Calculate completion time if available
            created = props.get('Created time', {}).get('created_time')
            last_edited = props.get('Last edited time', {}).get('last_edited_time')
            if created and last_edited:
                created_dt = datetime.fromisoformat(created.replace('Z', '+00:00'))
                edited_dt = datetime.fromisoformat(last_edited.replace('Z', '+00:00'))
                completion_times.append((edited_dt - created_dt).total_seconds() / 3600)

        elif status == 'In Progress':
            stats['in_progress'] += 1
        else:
            stats['backlog'] += 1

        # Priority
        priority = props.get('Priority', {}).get('select', {}).get('name')
        if priority in stats['by_priority']:
            stats['by_priority'][priority] += 1

    # Calculate averages
    if completion_times:
        stats['avg_completion_time'] = sum(completion_times) / len(completion_times)

    return stats

# Usage
stats = analyze_task_completion("daniel_kravtsov", days=30)
print(f"Completion rate: {stats['done'] / stats['total'] * 100:.1f}%")
print(f"Avg completion time: {stats['avg_completion_time']:.1f} hours")
```

## Pattern 7: Integration with Other Systems

### Use Case: Sync Jira tickets to Notion tasks

From: `algorithms/revenue_div/customer_success_dpt/tcs_team/jira_notion_integration/`

```python
from data_sources.notion.notion_cli import NotionAgent

def sync_jira_to_notion(jira_issue_key: str):
    """Create Notion task from Jira issue"""

    # Get Jira issue (pseudo-code - assumes Jira client exists)
    # jira_issue = jira_client.get_issue(jira_issue_key)

    agent = NotionAgent()

    # Map Jira fields to Notion
    priority_map = {
        'Highest': 'High',
        'High': 'High',
        'Medium': 'Medium',
        'Low': 'Low',
        'Lowest': 'Low'
    }

    status_map = {
        'To Do': 'In backlog',
        'In Progress': 'In Progress',
        'Done': 'Done'
    }

    # Create task
    task = agent.create_task(
        title=f"[{jira_issue_key}] {jira_issue.fields.summary}",
        description=jira_issue.fields.description,
        assignee=jira_issue.fields.assignee.emailAddress,
        status=status_map.get(jira_issue.fields.status.name, 'In backlog'),
        priority=priority_map.get(jira_issue.fields.priority.name, 'Medium')
    )

    # Add Jira link as comment
    from data_sources.notion.notion_client import NotionClient
    client = NotionClient()
    client.add_comment_to_page(
        page_id=task['id'],
        comment_text=f"Jira: https://jira.improvado.io/browse/{jira_issue_key}"
    )

    return task
```

## Pattern 8: Custom Database Queries

### Use Case: Advanced filtering with Notion API

```python
from data_sources.notion.notion_client import NotionClient

def query_tasks_with_custom_filter():
    """Advanced database query with complex filters"""

    client = NotionClient()

    # Complex filter: High priority OR assigned to me AND in progress
    filter_obj = {
        "or": [
            {
                "property": "Priority",
                "select": {
                    "equals": "High"
                }
            },
            {
                "and": [
                    {
                        "property": "Assignee",
                        "people": {
                            "contains": "NOTION_USER_ID"  # Daniel's ID
                        }
                    },
                    {
                        "property": "Status",
                        "status": {
                            "equals": "In Progress"
                        }
                    }
                ]
            }
        ]
    }

    # Sort by priority (High → Low) then by due date
    sorts = [
        {
            "property": "Priority",
            "direction": "descending"
        },
        {
            "property": "Due Date",
            "direction": "ascending"
        }
    ]

    results = client.query_database(
        database_id="42bffb6bf5354b828750be69024d374e",
        filter=filter_obj,
        sorts=sorts
    )

    return results
```

## Pattern 9: Automated Task Creation from Templates

### Use Case: Weekly recurring tasks, project templates

```python
from data_sources.notion.notion_cli import NotionAgent
from datetime import datetime, timedelta

def create_weekly_tasks(assignee: str):
    """Create weekly recurring tasks automatically"""

    agent = NotionAgent()

    # Weekly task template
    weekly_tasks = [
        {
            'title': 'Weekly team sync',
            'day': 0,  # Monday
            'priority': 'High'
        },
        {
            'title': 'Review dashboard metrics',
            'day': 2,  # Wednesday
            'priority': 'Medium'
        },
        {
            'title': 'Weekly report',
            'day': 4,  # Friday
            'priority': 'High'
        }
    ]

    # Calculate next occurrence for each task
    today = datetime.now()
    created_tasks = []

    for template in weekly_tasks:
        # Find next occurrence of this weekday
        days_ahead = template['day'] - today.weekday()
        if days_ahead <= 0:
            days_ahead += 7

        due_date = today + timedelta(days=days_ahead)

        task = agent.create_task(
            title=template['title'],
            assignee=assignee,
            priority=template['priority'],
            due_date=due_date.strftime('%Y-%m-%d'),
            status='In backlog'
        )

        created_tasks.append(task)

    return created_tasks

# Schedule weekly
# created_tasks = create_weekly_tasks("daniel_kravtsov")
```

## Pattern 10: Task Dependency Tracking

### Use Case: Track which tasks block others

```python
from data_sources.notion.notion_cli import NotionAgent
from data_sources.notion.notion_client import NotionClient

def find_blocking_tasks(task_id: str):
    """Find all tasks that this task depends on (blocks)"""

    client = NotionClient()
    agent = NotionAgent()

    # Get task tree (parents are blockers)
    tree = agent.get_task_tree(task_id, parent_depth=5, child_depth=0)

    blockers = []

    def extract_parents(node):
        """Recursively extract parent tasks"""
        parents = node.get('parents', [])
        for parent in parents:
            blockers.append({
                'id': parent['id'],
                'title': parent.get('title', 'Untitled'),
                'status': parent.get('status', 'Unknown')
            })
            extract_parents(parent)

    extract_parents(tree)

    # Check if any blockers are not done
    incomplete_blockers = [b for b in blockers if b['status'] != 'Done']

    return {
        'all_blockers': blockers,
        'incomplete_blockers': incomplete_blockers,
        'is_blocked': len(incomplete_blockers) > 0
    }

# Usage
blocking_info = find_blocking_tasks("abc123")
if blocking_info['is_blocked']:
    print("⚠️ Task is blocked by:")
    for blocker in blocking_info['incomplete_blockers']:
        print(f"  - {blocker['title']} ({blocker['status']})")
```

## Best Practices from Codebase

### 1. Always Use Try-Except for API Calls

```python
from data_sources.notion.notion_cli import NotionAgent

agent = NotionAgent()

try:
    task = agent.create_task(title="New task", assignee="daniel_kravtsov")
    print(f"✓ Created: {task['url']}")
except ValueError as e:
    print(f"✗ Validation error: {e}")
except Exception as e:
    print(f"✗ API error: {e}")
```

### 2. Cache User Lookups

```python
# Bad: Multiple API calls
for task_data in task_list:
    user_id = agent.get_user_id(task_data['assignee'])  # N API calls

# Good: Single API call + local lookup
all_users = agent.list_users()
user_map = {u['email']: u['id'] for u in all_users}

for task_data in task_list:
    user_id = user_map.get(task_data['assignee'])  # No API calls
```

### 3. Use Meaningful Task Titles

```python
# Bad
agent.create_task("Fix bug")

# Good
agent.create_task("Fix dashboard rendering bug for client XYZ - Metric cards not displaying")
```

### 4. Link Related Resources

```python
from data_sources.notion.notion_client import NotionClient

# After creating task, add links to related resources
client = NotionClient()
client.add_comment_to_page(
    page_id=task['id'],
    comment_text=f"""
Related resources:
- GitHub PR: https://github.com/org/repo/pull/123
- Jira: https://jira.company.com/browse/PROJ-456
- Design: https://figma.com/file/abc123
    """.strip()
)
```

### 5. Validate Before Bulk Operations

```python
def bulk_delete_tasks(task_ids: list):
    """Delete multiple tasks with validation"""

    agent = NotionAgent()
    client = NotionClient()

    # Validate all tasks exist and are deletable
    print("Validating tasks...")
    valid_tasks = []

    for task_id in task_ids:
        try:
            task = client.get_page(task_id)
            status = task['properties']['Status']['status']['name']

            if status == 'Done':
                print(f"⚠️ Skipping {task_id} (already Done)")
                continue

            valid_tasks.append(task_id)
        except Exception as e:
            print(f"✗ Invalid task {task_id}: {e}")

    # Confirm before deleting
    print(f"\nReady to delete {len(valid_tasks)} tasks")
    confirm = input("Continue? (yes/no): ")

    if confirm.lower() == 'yes':
        for task_id in valid_tasks:
            client.archive_page(task_id)
            print(f"✓ Deleted {task_id}")
```
