# Notion Operations Skill

A comprehensive Claude Code skill for managing Notion workspace operations through NotionAgent.

## Skill Overview

This skill enables Claude to autonomously:
- Create and manage Notion tasks
- Find and lookup users
- Navigate task hierarchies
- Query databases with advanced filters
- Handle authentication and permissions
- Troubleshoot common issues

## File Structure

```
.claude/skills/notion-operations/
├── SKILL.md                    # Main skill definition (Claude reads this first)
├── README.md                   # This file - documentation for humans
├── ADVANCED_PATTERNS.md        # Complex patterns from codebase
└── TROUBLESHOOTING.md          # Common issues and solutions
```

## How Claude Uses This Skill

### Automatic Activation

Claude will automatically invoke this skill when you ask about:
- "Create a Notion task for..."
- "Find user ID for..."
- "Get my tasks from Notion"
- "Show task tree for..."
- "Update task status to..."

### Progressive Disclosure

The skill uses a tiered structure:

1. **SKILL.md** - Core operations and common use cases
   - Read first by Claude for most operations
   - Contains authentication setup, basic CRUD, common patterns

2. **ADVANCED_PATTERNS.md** - Complex workflows
   - Referenced for hierarchical task management
   - Bulk operations and analytics
   - Integration with other systems

3. **TROUBLESHOOTING.md** - Error resolution
   - Referenced when operations fail
   - Diagnostic procedures
   - Performance optimization

## Quick Start for Humans

### Prerequisites

1. **Notion Integration Token:**
   ```bash
   # Get token from https://www.notion.so/my-integrations
   echo 'NOTION_TOKEN="secret_abc123..."' >> .env
   ```

2. **Optional: Auto-assignment:**
   ```bash
   echo 'NOTION_YOUR_EMAIL="your.email@domain.com"' >> .env
   ```

### Test the Skill

Ask Claude:

```
Create a Notion task titled "Test skill functionality" assigned to me
```

Claude will:
1. Activate the `notion-operations` skill
2. Load authentication instructions from SKILL.md
3. Find your user ID (using NOTION_YOUR_EMAIL)
4. Create the task using NotionAgent
5. Return the task URL

## Common Usage Patterns

### 1. Task Management

**User asks:** "Create a high priority task for fixing the dashboard bug, assign to Daniel, due next Friday"

**Claude does:**
- Activates skill
- Finds Daniel's user ID
- Calculates next Friday's date
- Creates task with priority="High"
- Returns task URL

### 2. User Lookup

**User asks:** "What's Nataliia's Notion user ID?"

**Claude does:**
- Uses find-user command
- Returns ID: `NOTION_USER_ID_2`

### 3. Task Queries

**User asks:** "Show me all high priority tasks in progress"

**Claude does:**
- Queries tasks with filters
- Formats results for display
- Shows task URLs

### 4. Hierarchy Navigation

**User asks:** "Show me the full task tree for this Notion URL"

**Claude does:**
- Extracts page ID from URL
- Builds bidirectional tree
- Formats with parent/child relationships
- Shows statistics

## Integration with Sub-Agents

This skill complements the Task tool's notion-agent subagent:

**Skill (this):** Quick operations within main conversation
- "Create a task"
- "Find a user"
- Simple queries

**Sub-agent:** Complex multi-step operations
- Bulk task management
- Advanced hierarchy analysis
- Integration workflows

Example delegation:

```
User: "Analyze all my tasks and create a weekly report"

Claude: This requires complex analysis. I'll use the Task tool
        with notion-agent subagent for this operation.
```

## Best Practices

### 1. Skill Activation is Automatic

✅ **Good:** "Create a Notion task for code review"
- Claude automatically activates skill

❌ **Unnecessary:** "Use the Notion skill to create a task"
- Explicit skill invocation not needed

### 2. Provide Context

✅ **Good:** "Create task assigned to daniel_kravtsov with high priority"
- Provides all needed parameters

❌ **Vague:** "Make a task"
- Claude will ask for missing info

### 3. Trust Error Handling

The skill includes comprehensive troubleshooting:
- Authentication errors → checks .env
- User not found → suggests find-user
- API errors → provides diagnostic steps

## Skill Maintenance

### Adding New Patterns

When you develop new Notion workflows:

1. **Test the workflow** in Python/CLI
2. **Document in ADVANCED_PATTERNS.md:**
   ```markdown
   ## Pattern X: Your Pattern Name

   ### Use Case: When to use this

   ```python
   # Code example
   ```
   ```
3. **Update SKILL.md** if it's a common operation

### Updating for API Changes

If Notion API changes:

1. **Update notion_client.py** in codebase
2. **Update SKILL.md** with new parameters/methods
3. **Add to TROUBLESHOOTING.md** if breaking change
4. **Update version history** in SKILL.md

## Testing the Skill

### Manual Testing

Ask Claude these test cases:

1. **Authentication:**
   - "Check if my Notion token is valid"

2. **Task Creation:**
   - "Create a test task"
   - "Create task with description, priority, and due date"

3. **User Management:**
   - "List all Notion users"
   - "Find Daniel Kravtsov's user ID"

4. **Queries:**
   - "Show my tasks from last week"
   - "Get all high priority tasks"

5. **Tree Navigation:**
   - "Show task tree for [Notion URL]"

### Automated Testing

Run the test script:

```bash
python3 << 'EOF'
from data_sources.notion.notion_cli import NotionAgent

agent = NotionAgent()

# Test 1: Authentication
print("Test 1: Authentication...")
try:
    users = agent.list_users()
    print(f"✓ Connected ({len(users)} users)")
except Exception as e:
    print(f"✗ Failed: {e}")

# Test 2: User lookup
print("\nTest 2: User lookup...")
try:
    user_id = agent.get_user_id("daniel_kravtsov")
    print(f"✓ Found user: {user_id}")
except Exception as e:
    print(f"✗ Failed: {e}")

# Test 3: Task creation
print("\nTest 3: Task creation...")
try:
    task = agent.create_task(
        title="[TEST] Skill validation",
        assignee="daniel_kravtsov",
        status="In backlog"
    )
    print(f"✓ Created: {task['url']}")

    # Cleanup
    from data_sources.notion.notion_client import NotionClient
    client = NotionClient()
    client.archive_page(task['id'])
    print("✓ Cleaned up test task")
except Exception as e:
    print(f"✗ Failed: {e}")

print("\n✅ All tests completed")
EOF
```

## Troubleshooting

### Skill Not Activating

**Problem:** Claude doesn't use the skill when expected.

**Diagnosis:**
1. Check skill description in SKILL.md is clear
2. Verify `.claude/skills/notion-operations/` exists
3. Restart Claude Code if recently added

**Solution:**
- Make request more explicit: "Create a Notion task..."
- Update description in SKILL.md to include trigger keywords

### Skill File Not Found

**Problem:** Claude can't read ADVANCED_PATTERNS.md or TROUBLESHOOTING.md

**Solution:**
- Ensure files are in same directory as SKILL.md
- Check file permissions: `ls -la .claude/skills/notion-operations/`
- Reference files explicitly in SKILL.md with relative paths

### Outdated Instructions

**Problem:** Skill uses old API patterns.

**Solution:**
- Update SKILL.md with new patterns from notion_client.py
- Test changes manually before committing
- Update version history

## Related Documentation

- **Main codebase:** `data_sources/notion/`
- **NotionAgent README:** `data_sources/notion/README.md`
- **Zero Inbox workflow:** `data_sources/notion/00_zero_inbox/`
- **Jira integration:** `algorithms/revenue_div/customer_success_dpt/tcs_team/jira_notion_integration/`

## Version History

- **v1.0 (2025-01-15):**
  - Initial skill creation
  - Core CRUD operations
  - User management
  - Task tree navigation
  - Authentication and troubleshooting guides

## Contributing

When enhancing this skill:

1. **Test locally** with Python/CLI first
2. **Document in appropriate file:**
   - Common operations → SKILL.md
   - Complex patterns → ADVANCED_PATTERNS.md
   - Error fixes → TROUBLESHOOTING.md
3. **Update version history** in SKILL.md and this README
4. **Test with Claude** to ensure activation works
5. **Commit with clear message:**
   ```bash
   git add .claude/skills/notion-operations/
   git commit -m "feat(notion-skill): Add [feature description]"
   ```

## Support

For issues with:
- **Skill activation:** Check SKILL.md description field
- **API errors:** See TROUBLESHOOTING.md
- **Missing patterns:** Add to ADVANCED_PATTERNS.md
- **Codebase issues:** Update `data_sources/notion/`

## License

This skill is part of the chrome-extension-tcs project and follows the same license.
