# Notion Operations Troubleshooting Guide

Common issues and solutions when working with NotionAgent.

## Authentication Issues

### Error: "NOTION_TOKEN не найден"

**Cause:** `NOTION_TOKEN` environment variable not set.

**Solution:**
```bash
# Add to .env file
echo 'NOTION_TOKEN="secret_abc123..."' >> .env

# Or export in terminal
export NOTION_TOKEN="secret_abc123..."

# Verify
python -c "import os; from dotenv import load_dotenv; load_dotenv(); print(os.getenv('NOTION_TOKEN'))"
```

### Error: 401 Unauthorized

**Causes:**
1. Token is invalid or expired
2. Integration not installed in workspace
3. Token copied incorrectly (with quotes or spaces)

**Solutions:**

```bash
# 1. Verify token format (should start with "secret_")
echo $NOTION_TOKEN

# 2. Regenerate token
# - Go to https://www.notion.so/my-integrations
# - Click your integration
# - Regenerate token
# - Update .env file

# 3. Verify integration is installed
# - Go to any Notion page
# - Click "..." → "Add connections"
# - Ensure your integration is listed and enabled

# 4. Test token programmatically
python3 << 'EOF'
from data_sources.notion.notion_client import NotionClient
try:
    client = NotionClient()
    client.verify_token()
    print("✓ Token is valid")
except Exception as e:
    print(f"✗ Token error: {e}")
EOF
```

### Error: "Cannot auto-assign task: NOTION_YOUR_EMAIL not set"

**Cause:** Trying to create task without `assignee` parameter and `NOTION_YOUR_EMAIL` not configured.

**Solutions:**

```bash
# Option 1: Set NOTION_YOUR_EMAIL
echo 'NOTION_YOUR_EMAIL="your.email@domain.com"' >> .env

# Option 2: Always specify assignee
python -m data_sources.notion.notion_cli create-task "Task title" --assignee daniel_kravtsov

# Option 3: Find your user ID and use it directly
python -m data_sources.notion.notion_cli find-user "your.email@domain.com" --show-id
```

## User Management Issues

### Error: "User 'xyz' not found"

**Cause:** Username/email doesn't match any user in workspace.

**Diagnosis:**

```bash
# List all users to see available options
python -m data_sources.notion.notion_cli list-users

# Search for specific user
python -m data_sources.notion.notion_cli find-user "kravtsov"

# Try different search terms
python -m data_sources.notion.notion_cli find-user "daniel"
python -m data_sources.notion.notion_cli find-user "daniel@improvado.io"
```

**Solutions:**

```python
from data_sources.notion.notion_cli import NotionAgent

agent = NotionAgent()

# Get all users
users = agent.list_users()
for user in users:
    print(f"Name: {user['name']}, Email: {user['email']}, ID: {user['id']}")

# Use exact email or ID
task = agent.create_task(
    title="Task",
    assignee="daniel@improvado.io"  # Or user ID directly
)
```

### Multiple Users with Similar Names

**Problem:** Search returns first match, but you want a different user.

**Solution:**

```python
# Use exact email instead of name
users = agent.list_users(search="daniel@improvado.io")

# Or get all and filter manually
all_users = agent.list_users()
daniel_users = [u for u in all_users if 'daniel' in u['name'].lower()]

for u in daniel_users:
    print(f"{u['name']} - {u['email']}")

# Use specific ID
agent.create_task(title="Task", assignee="NOTION_USER_ID")
```

## Task Creation Issues

### Error: 404 Not Found - Database not found

**Cause:** Database ID incorrect or database not shared with integration.

**Diagnosis:**

```bash
# Check default database ID
python3 << 'EOF'
from data_sources.notion.agent_commands.task_commands import TASKS_DB_ID
print(f"Default database: {TASKS_DB_ID}")
EOF
```

**Solutions:**

1. **Share database with integration:**
   - Open tasks database in Notion
   - Click "..." → "Add connections"
   - Add your integration

2. **Find correct database ID:**
   ```python
   from data_sources.notion.notion_client import NotionClient

   client = NotionClient()

   # Search for database
   results = client.search("Tasks")

   for result in results:
       if result.get('object') == 'database':
           print(f"Database: {result['title'][0]['plain_text']}")
           print(f"ID: {result['id']}")
   ```

3. **Use custom database ID:**
   ```python
   task = agent.create_task(
       title="Task",
       database_id="your_custom_db_id"
   )
   ```

### Error: 400 Bad Request - validation_error

**Causes:**
1. Invalid property values
2. Property doesn't exist in database
3. Wrong property type

**Common validation errors:**

```python
# ❌ Invalid status value
agent.create_task(title="Task", status="InvalidStatus")
# Solution: Use valid status from database schema

# ❌ Invalid priority value
agent.create_task(title="Task", priority="Super High")
# Solution: Use "High", "Medium", or "Low"

# ❌ Invalid date format
agent.create_task(title="Task", due_date="2025/01/15")
# Solution: Use ISO format "2025-01-15"

# ❌ Invalid parent task ID
agent.create_task(title="Task", parent_task="invalid-id")
# Solution: Verify parent task exists and is accessible
```

**Debugging property values:**

```python
from data_sources.notion.notion_client import NotionClient

client = NotionClient()

# Get database schema
db = client.get_database("42bffb6bf5354b828750be69024d374e")

# Check available options for select properties
for prop_name, prop_config in db['properties'].items():
    if prop_config['type'] == 'select':
        print(f"\n{prop_name} options:")
        for option in prop_config['select']['options']:
            print(f"  - {option['name']}")
    elif prop_config['type'] == 'status':
        print(f"\n{prop_name} options:")
        for option in prop_config['status']['options']:
            print(f"  - {option['name']}")
```

## Task Query Issues

### No Results Returned

**Causes:**
1. Filters too restrictive
2. Database not shared with integration
3. No tasks match criteria

**Diagnosis:**

```python
from data_sources.notion.notion_cli import NotionAgent

agent = NotionAgent()

# Test with no filters
all_tasks = agent.get_tasks()
print(f"Total tasks accessible: {len(all_tasks)}")

# Test with individual filters
by_assignee = agent.get_tasks(assignee="daniel_kravtsov")
print(f"By assignee: {len(by_assignee)}")

by_status = agent.get_tasks(status="In Progress")
print(f"By status: {len(by_status)}")

# Check if issue is with days_back filter
recent = agent.get_tasks(days_back=7)
print(f"Last 7 days: {len(recent)}")

older = agent.get_tasks(days_back=30)
print(f"Last 30 days: {len(older)}")
```

**Solutions:**

```python
# Broaden search criteria
tasks = agent.get_tasks(days_back=90)  # Increase time window

# Remove filters one by one to identify issue
tasks = agent.get_tasks(assignee="daniel_kravtsov")  # Works?
tasks = agent.get_tasks(assignee="daniel_kravtsov", status="In Progress")  # Fails?
# → Issue is with status filter

# Verify property names match database
from data_sources.notion.notion_client import NotionClient
client = NotionClient()
db = client.get_database("42bffb6bf5354b828750be69024d374e")
print("Available properties:", list(db['properties'].keys()))
```

### Slow Query Performance

**Cause:** Large database with no pagination.

**Solution:**

```python
from data_sources.notion.notion_client import NotionClient

client = NotionClient()

# Use pagination for large datasets
def get_all_tasks_paginated(database_id: str):
    """Get all tasks with pagination"""
    all_results = []
    has_more = True
    next_cursor = None

    while has_more:
        response = client.query_database(
            database_id=database_id,
            start_cursor=next_cursor,
            page_size=100  # Max allowed
        )

        all_results.extend(response['results'])
        has_more = response['has_more']
        next_cursor = response.get('next_cursor')

        print(f"Fetched {len(all_results)} tasks...")

    return all_results

# Usage
tasks = get_all_tasks_paginated("42bffb6bf5354b828750be69024d374e")
```

## Task Tree Issues

### Error: Maximum recursion depth exceeded

**Cause:** Circular reference in task hierarchy (task is its own ancestor).

**Diagnosis:**

```python
from data_sources.notion.notion_cli import NotionAgent

agent = NotionAgent()

# Check for circular references
def detect_circular_reference(task_id: str):
    """Detect if task has circular parent reference"""
    visited = set()
    current_id = task_id

    client = NotionClient()

    for _ in range(100):  # Max depth limit
        if current_id in visited:
            return f"Circular reference detected at {current_id}"

        visited.add(current_id)

        # Get parent
        page = client.get_page(current_id)
        parent_rel = page['properties'].get('Parent-task', {}).get('relation', [])

        if not parent_rel:
            return "No circular reference"

        current_id = parent_rel[0]['id']

    return "Max depth reached"

result = detect_circular_reference("abc123")
print(result)
```

**Solution:**

```python
# Fix circular reference manually
from data_sources.notion.notion_client import NotionClient

client = NotionClient()

# Remove parent relationship
client.update_page_properties(
    page_id="abc123",
    properties={
        "Parent-task": {"relation": []}  # Clear parent
    }
)
```

### Tree Missing Expected Tasks

**Causes:**
1. `parent_depth` or `child_depth` too low
2. Tasks not properly linked
3. Tasks archived/deleted

**Solution:**

```python
agent = NotionAgent()

# Increase depth limits
tree = agent.get_task_tree(
    "abc123",
    parent_depth=10,   # Default: 3
    child_depth=20     # Default: 10
)

# Check statistics
stats = agent.get_tree_statistics(tree)
print(f"Total nodes: {stats['total_nodes']}")
print(f"Max depth: {stats['max_depth']}")

# If max_depth equals your limit, increase further
if stats['max_depth'] >= 20:
    tree = agent.get_task_tree("abc123", child_depth=50)
```

## API Rate Limiting

### Error: 429 Too Many Requests

**Cause:** Exceeded Notion API rate limits (3 requests/second).

**Solution:**

```python
import time
from data_sources.notion.notion_cli import NotionAgent

agent = NotionAgent()

# Add delays between bulk operations
task_titles = ["Task 1", "Task 2", "Task 3", ...]

created = []
for i, title in enumerate(task_titles):
    task = agent.create_task(title=title)
    created.append(task)

    # Rate limit: 3 req/sec → 0.33s between requests
    if (i + 1) % 3 == 0:
        time.sleep(1)

print(f"Created {len(created)} tasks")
```

**Better solution with async:**

```python
import asyncio
from data_sources.notion.notion_client import NotionClient

async def create_tasks_with_rate_limit(task_titles: list):
    """Create tasks respecting rate limits"""
    client = NotionClient()

    tasks = []
    for i, title in enumerate(task_titles):
        task = client.create_page(...)
        tasks.append(task)

        # Rate limit
        if (i + 1) % 3 == 0:
            await asyncio.sleep(1)

    return tasks
```

## Debugging Tips

### Enable Verbose Logging

```python
import logging

# Enable debug logging for Notion client
logging.basicConfig(level=logging.DEBUG)

from data_sources.notion.notion_cli import NotionAgent
agent = NotionAgent()

# Now all API requests will be logged
tasks = agent.get_tasks()
```

### Inspect Raw API Response

```python
from data_sources.notion.notion_client import NotionClient

client = NotionClient()

# Get raw response
response = client.get_page("abc123")

# Pretty print
import json
print(json.dumps(response, indent=2))
```

### Test Connection

```python
from data_sources.notion.notion_client import NotionClient

def test_notion_connection():
    """Test Notion API connection and permissions"""

    try:
        client = NotionClient()
        print("✓ Client initialized")

        # Test token
        client.verify_token()
        print("✓ Token valid")

        # Test database access
        db = client.get_database("42bffb6bf5354b828750be69024d374e")
        print(f"✓ Database accessible: {db['title'][0]['plain_text']}")

        # Test user access
        users = client.get_users()
        print(f"✓ Users accessible: {len(users)} users")

        # Test search
        results = client.search("test")
        print(f"✓ Search working: {len(results['results'])} results")

        print("\n✅ All tests passed!")
        return True

    except Exception as e:
        print(f"\n❌ Test failed: {e}")
        return False

# Run tests
test_notion_connection()
```

## Common Workflow Issues

### Task Not Showing in Notion UI

**Causes:**
1. Task created in wrong database
2. Task filtered out by view settings
3. Task created but not visible due to permissions

**Solutions:**

```python
from data_sources.notion.notion_cli import NotionAgent

agent = NotionAgent()

# Get task URL and open in browser
task = agent.create_task(title="Test Task")
print(f"Task URL: {task['url']}")
# Open this URL in browser to verify

# Check task properties
from data_sources.notion.notion_client import NotionClient
client = NotionClient()

page = client.get_page(task['id'])
print("Properties:")
for prop_name, prop_value in page['properties'].items():
    print(f"  {prop_name}: {prop_value}")
```

### Cannot Update Task

**Causes:**
1. Using wrong page ID
2. Property name typo
3. Invalid property value

**Diagnosis:**

```python
# Verify page exists
page = client.get_page("abc123")
print(f"Page title: {page['properties']['title']['title'][0]['text']['content']}")

# Check current properties
for prop_name in page['properties'].keys():
    print(f"- {prop_name}")

# Attempt update with error handling
try:
    agent.update_task_status("abc123", "Done")
    print("✓ Update successful")
except Exception as e:
    print(f"✗ Update failed: {e}")
    print(f"Error type: {type(e).__name__}")
```

## Performance Optimization

### Slow Task Creation

```python
# ❌ Slow: Multiple user lookups
for task_data in task_list:
    user_id = agent.get_user_id(task_data['assignee'])
    agent.create_task(title=task_data['title'], assignee=user_id)

# ✅ Fast: Cache user lookups
users = agent.list_users()
user_map = {u['email']: u['id'] for u in users}

for task_data in task_list:
    user_id = user_map.get(task_data['assignee'])
    agent.create_task(title=task_data['title'], assignee=user_id)
```

### Memory Issues with Large Trees

```python
# ❌ Memory heavy: Load entire tree
tree = agent.get_task_tree("abc123", child_depth=100)

# ✅ Memory efficient: Iterative loading
def get_tree_iteratively(root_id: str, max_depth: int = 100):
    """Load tree level by level to avoid memory issues"""
    client = NotionClient()

    nodes_by_level = {0: [root_id]}
    processed = set()

    for level in range(max_depth):
        if level not in nodes_by_level:
            break

        next_level = []
        for node_id in nodes_by_level[level]:
            if node_id in processed:
                continue

            page = client.get_page(node_id)
            # Process page...

            # Get children
            children_rel = page['properties'].get('Sub-task', {}).get('relation', [])
            next_level.extend([c['id'] for c in children_rel])

            processed.add(node_id)

        if next_level:
            nodes_by_level[level + 1] = next_level

    return processed

# Usage
all_task_ids = get_tree_iteratively("abc123")
```

## Getting Help

If issues persist:

1. **Check Notion API Status:** https://status.notion.so/
2. **Review API Docs:** https://developers.notion.com/reference/intro
3. **Test with curl:**
   ```bash
   curl -X GET https://api.notion.com/v1/users/me \
     -H "Authorization: Bearer $NOTION_TOKEN" \
     -H "Notion-Version: 2022-06-28"
   ```
4. **Check codebase examples:** `data_sources/notion/00_zero_inbox/`
5. **Enable debug mode and share logs**
