#!/bin/bash
# Test script for Notion Operations Skill
# Run this to verify skill is properly configured

set -e

echo "🔍 Testing Notion Operations Skill..."
echo ""

# Test 1: Check skill files exist
echo "Test 1: Checking skill files..."
if [ -f ".claude/skills/notion-tasks-operations/SKILL.md" ]; then
    echo "✓ SKILL.md exists"
else
    echo "✗ SKILL.md not found"
    exit 1
fi

if [ -f ".claude/skills/notion-tasks-operations/ADVANCED_PATTERNS.md" ]; then
    echo "✓ ADVANCED_PATTERNS.md exists"
else
    echo "✗ ADVANCED_PATTERNS.md not found"
    exit 1
fi

if [ -f ".claude/skills/notion-tasks-operations/TROUBLESHOOTING.md" ]; then
    echo "✓ TROUBLESHOOTING.md exists"
else
    echo "✗ TROUBLESHOOTING.md not found"
    exit 1
fi

# Test 2: Validate YAML frontmatter
echo ""
echo "Test 2: Validating YAML frontmatter..."
if head -1 .claude/skills/notion-tasks-operations/SKILL.md | grep -q "^---$"; then
    echo "✓ YAML opening delimiter found"
else
    echo "✗ YAML opening delimiter not found"
    exit 1
fi

if head -4 .claude/skills/notion-tasks-operations/SKILL.md | tail -1 | grep -q "^---$"; then
    echo "✓ YAML closing delimiter found"
else
    echo "✗ YAML closing delimiter not found"
    exit 1
fi

# Test 3: Check required fields
echo ""
echo "Test 3: Checking required fields..."
if grep -q "^name: notion-tasks-operations$" .claude/skills/notion-tasks-operations/SKILL.md; then
    echo "✓ Skill name is valid"
else
    echo "✗ Skill name invalid or missing"
    exit 1
fi

if grep -q "^description:" .claude/skills/notion-tasks-operations/SKILL.md; then
    echo "✓ Description field exists"
else
    echo "✗ Description field missing"
    exit 1
fi

# Test 4: Check Python dependencies
echo ""
echo "Test 4: Checking Python dependencies..."
if python3 -c "from data_sources.notion.notion_cli import NotionAgent" 2>/dev/null; then
    echo "✓ NotionAgent importable"
else
    echo "✗ Cannot import NotionAgent"
    exit 1
fi

# Test 5: Check environment setup
echo ""
echo "Test 5: Checking environment configuration..."
if [ -n "$NOTION_TOKEN" ]; then
    echo "✓ NOTION_TOKEN is set"
else
    echo "⚠️  NOTION_TOKEN not set (required for actual operations)"
fi

if [ -n "$NOTION_YOUR_EMAIL" ]; then
    echo "✓ NOTION_YOUR_EMAIL is set"
else
    echo "⚠️  NOTION_YOUR_EMAIL not set (optional, but recommended)"
fi

# Test 6: Verify skill structure
echo ""
echo "Test 6: Verifying skill structure..."
total_lines=$(wc -l < .claude/skills/notion-tasks-operations/SKILL.md)
if [ "$total_lines" -gt 100 ]; then
    echo "✓ SKILL.md has substantial content ($total_lines lines)"
else
    echo "✗ SKILL.md seems too short ($total_lines lines)"
    exit 1
fi

# Test 7: Check for common patterns
echo ""
echo "Test 7: Checking for required content..."
if grep -q "create_task" .claude/skills/notion-tasks-operations/SKILL.md; then
    echo "✓ Task creation documented"
else
    echo "✗ Task creation not documented"
    exit 1
fi

if grep -q "get_user_id\|find-user" .claude/skills/notion-tasks-operations/SKILL.md; then
    echo "✓ User management documented"
else
    echo "✗ User management not documented"
    exit 1
fi

if grep -q "get_task_tree\|task tree" .claude/skills/notion-tasks-operations/SKILL.md; then
    echo "✓ Tree navigation documented"
else
    echo "✗ Tree navigation not documented"
    exit 1
fi

# Summary
echo ""
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
echo "✅ All tests passed!"
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
echo ""
echo "Skill Statistics:"
echo "  - SKILL.md: $(wc -l < .claude/skills/notion-tasks-operations/SKILL.md) lines"
echo "  - ADVANCED_PATTERNS.md: $(wc -l < .claude/skills/notion-tasks-operations/ADVANCED_PATTERNS.md) lines"
echo "  - TROUBLESHOOTING.md: $(wc -l < .claude/skills/notion-tasks-operations/TROUBLESHOOTING.md) lines"
echo "  - Total: $(cat .claude/skills/notion-tasks-operations/*.md | wc -l) lines"
echo ""
echo "To use this skill, simply ask Claude:"
echo '  "Create a Notion task titled '\''Test task'\''"'
echo ""
echo "Claude will automatically activate the notion-tasks-operations skill!"
