"""
Created by: Claude Code
Session ID: 2025-12-03 (notion-skill-tests)
Date: 2025-12-03
Purpose: Tests for notion-tasks-operations skill functionality

Tests cover:
1. Block ID extraction from URL anchors
2. append_blocks_after() function with mocked API
3. NotionClient.append_block_children() with mocked API
4. Mermaid block creation
"""

import os
import sys
import unittest
from unittest.mock import patch, MagicMock
import json

# Add paths for imports
sys.path.insert(0, os.path.dirname(os.path.dirname(os.path.dirname(os.path.dirname(os.path.dirname(os.path.abspath(__file__)))))))


class TestBlockIdExtraction(unittest.TestCase):
    """Test block ID extraction from Notion URL anchors."""

    def test_extract_block_id_from_anchor_32_chars(self):
        """Test converting 32-char anchor to 36-char UUID format."""
        # URL anchor format (no dashes, 32 chars)
        anchor = "2bf9aec6212580fab178ff9f9872e0f9"

        # Convert to API format (with dashes, 36 chars)
        block_id = f"{anchor[:8]}-{anchor[8:12]}-{anchor[12:16]}-{anchor[16:20]}-{anchor[20:]}"

        expected = "2bf9aec6-2125-80fa-b178-ff9f9872e0f9"
        self.assertEqual(block_id, expected)
        self.assertEqual(len(block_id), 36)

    def test_extract_block_id_preserves_case(self):
        """Test that block ID extraction preserves lowercase hex chars."""
        anchor = "abcdef12345678901234567890abcdef"
        block_id = f"{anchor[:8]}-{anchor[8:12]}-{anchor[12:16]}-{anchor[16:20]}-{anchor[20:]}"

        self.assertTrue(block_id.islower() or block_id.replace('-', '').isalnum())

    def test_extract_page_id_from_url(self):
        """Test extracting page ID from full Notion URL."""
        url = "https://www.notion.so/improvado-home/AI-Agent-Call-2be9aec621258070b6dae3699106662e"

        # Extract last 32 chars before any query params or anchors
        url_clean = url.split('?')[0].split('#')[0]
        page_id_raw = url_clean[-32:]

        # Convert to UUID format
        page_id = f"{page_id_raw[:8]}-{page_id_raw[8:12]}-{page_id_raw[12:16]}-{page_id_raw[16:20]}-{page_id_raw[20:]}"

        expected = "2be9aec6-2125-8070-b6da-e3699106662e"
        self.assertEqual(page_id, expected)

    def test_extract_anchor_from_url_with_hash(self):
        """Test extracting anchor block ID from URL with hash."""
        url = "https://www.notion.so/page-2be9aec621258070b6dae3699106662e#2bf9aec6212580fab178ff9f9872e0f9"

        # Extract anchor
        if '#' in url:
            anchor = url.split('#')[1]
        else:
            anchor = None

        self.assertEqual(anchor, "2bf9aec6212580fab178ff9f9872e0f9")
        self.assertEqual(len(anchor), 32)


class TestMermaidBlockCreation(unittest.TestCase):
    """Test Mermaid diagram block creation for Notion API."""

    def test_create_mermaid_block(self):
        """Test creating a Mermaid code block."""
        mermaid_code = """graph LR
    A[Start] --> B[End]"""

        block = {
            "type": "code",
            "code": {
                "language": "mermaid",
                "rich_text": [
                    {
                        "type": "text",
                        "text": {"content": mermaid_code}
                    }
                ]
            }
        }

        self.assertEqual(block["type"], "code")
        self.assertEqual(block["code"]["language"], "mermaid")
        self.assertEqual(block["code"]["rich_text"][0]["text"]["content"], mermaid_code)

    def test_create_mermaid_block_with_caption(self):
        """Test creating Mermaid block with caption."""
        mermaid_code = "graph TD\n    A --> B"

        block = {
            "type": "code",
            "code": {
                "language": "mermaid",
                "rich_text": [{"type": "text", "text": {"content": mermaid_code}}],
                "caption": [{"type": "text", "text": {"content": "Pipeline Flow"}}]
            }
        }

        self.assertEqual(block["code"]["caption"][0]["text"]["content"], "Pipeline Flow")


class TestAppendBlocksAfter(unittest.TestCase):
    """Test append_blocks_after() function with mocked Notion API."""

    @patch('requests.patch')
    def test_append_blocks_after_success(self, mock_patch):
        """Test successful block insertion after specific block."""
        # Mock successful API response
        mock_response = MagicMock()
        mock_response.status_code = 200
        mock_response.json.return_value = {
            "results": [
                {"id": "new-block-1", "type": "divider"},
                {"id": "new-block-2", "type": "heading_2"}
            ]
        }
        mock_patch.return_value = mock_response

        # Import function (inline to avoid import errors)
        import requests

        def append_blocks_after(page_id: str, after_block_id: str, blocks: list) -> dict:
            token = os.getenv('NOTION_TOKEN', 'test-token')
            headers = {
                "Authorization": f"Bearer {token}",
                "Content-Type": "application/json",
                "Notion-Version": "2022-06-28"
            }
            url = f"https://api.notion.com/v1/blocks/{page_id}/children"
            data = {"children": blocks, "after": after_block_id}
            response = requests.patch(url, headers=headers, json=data)
            return response.json()

        # Test data
        page_id = "2be9aec6-2125-8070-b6da-e3699106662e"
        after_block_id = "2bf9aec6-2125-80fa-b178-ff9f9872e0f9"
        blocks = [
            {"type": "divider", "divider": {}},
            {"type": "heading_2", "heading_2": {"rich_text": [{"type": "text", "text": {"content": "Test"}}]}}
        ]

        result = append_blocks_after(page_id, after_block_id, blocks)

        # Verify API was called with correct params
        mock_patch.assert_called_once()
        call_args = mock_patch.call_args

        # Check URL
        self.assertIn(page_id, call_args[0][0])

        # Check body has 'after' parameter
        call_json = call_args[1]['json']
        self.assertEqual(call_json['after'], after_block_id)
        self.assertEqual(len(call_json['children']), 2)

        # Check result
        self.assertEqual(len(result['results']), 2)

    @patch('requests.patch')
    def test_append_blocks_after_api_error(self, mock_patch):
        """Test handling API error response."""
        mock_response = MagicMock()
        mock_response.status_code = 404
        mock_response.text = "Block not found"
        mock_response.raise_for_status.side_effect = Exception("404 Not Found")
        mock_patch.return_value = mock_response

        import requests

        def append_blocks_after(page_id: str, after_block_id: str, blocks: list) -> dict:
            token = os.getenv('NOTION_TOKEN', 'test-token')
            headers = {"Authorization": f"Bearer {token}", "Notion-Version": "2022-06-28"}
            url = f"https://api.notion.com/v1/blocks/{page_id}/children"
            response = requests.patch(url, headers=headers, json={"children": blocks, "after": after_block_id})
            response.raise_for_status()
            return response.json()

        with self.assertRaises(Exception):
            append_blocks_after("invalid-page", "invalid-block", [])


class TestNotionClientIntegration(unittest.TestCase):
    """Test NotionClient methods with mocked API."""

    def setUp(self):
        """Set up test fixtures."""
        # Mock environment
        os.environ['NOTION_TOKEN'] = 'test-token-for-testing'

    @patch('requests.get')
    @patch('requests.patch')
    def test_append_block_children_method(self, mock_patch, mock_get):
        """Test NotionClient.append_block_children() method."""
        # Mock verify_token (GET request)
        mock_get_response = MagicMock()
        mock_get_response.status_code = 200
        mock_get_response.json.return_value = {"id": "bot-id", "type": "bot", "name": "Test Bot"}
        mock_get.return_value = mock_get_response

        # Mock append (PATCH request)
        mock_patch_response = MagicMock()
        mock_patch_response.status_code = 200
        mock_patch_response.json.return_value = {"results": [{"id": "new-block"}]}
        mock_patch.return_value = mock_patch_response

        # Import NotionClient
        from data_sources.notion.notion_client import NotionClient

        client = NotionClient(token='test-token')

        blocks = [{"type": "paragraph", "paragraph": {"rich_text": [{"type": "text", "text": {"content": "Test"}}]}}]

        result = client.append_block_children("page-id", blocks)

        self.assertIn("results", result)
        mock_patch.assert_called_once()


class TestCalloutBlockCreation(unittest.TestCase):
    """Test callout block creation with GitHub links."""

    def test_create_callout_with_link(self):
        """Test creating callout block with clickable link."""
        github_url = "https://github.com/tekliner/tcs-chrome-extension"

        block = {
            "type": "callout",
            "callout": {
                "rich_text": [
                    {"type": "text", "text": {"content": "📂 Source Code: "}},
                    {
                        "type": "text",
                        "text": {"content": "GitHub Repository", "link": {"url": github_url}},
                        "annotations": {"bold": True, "color": "blue"}
                    }
                ],
                "icon": {"type": "emoji", "emoji": "💻"},
                "color": "gray_background"
            }
        }

        self.assertEqual(block["type"], "callout")
        self.assertEqual(block["callout"]["icon"]["emoji"], "💻")
        self.assertEqual(block["callout"]["rich_text"][1]["text"]["link"]["url"], github_url)


class TestDividerBlock(unittest.TestCase):
    """Test divider block creation."""

    def test_create_divider_block(self):
        """Test creating simple divider block."""
        block = {"type": "divider", "divider": {}}

        self.assertEqual(block["type"], "divider")
        self.assertEqual(block["divider"], {})


class TestHeadingBlocks(unittest.TestCase):
    """Test heading block creation."""

    def test_create_heading_2_block(self):
        """Test creating H2 heading block."""
        block = {
            "type": "heading_2",
            "heading_2": {
                "rich_text": [{"type": "text", "text": {"content": "🏗️ Skill Architecture"}}]
            }
        }

        self.assertEqual(block["type"], "heading_2")
        self.assertEqual(block["heading_2"]["rich_text"][0]["text"]["content"], "🏗️ Skill Architecture")


class TestTableExtraction(unittest.TestCase):
    """Test table extraction utilities."""

    def test_extract_ids_from_url_with_anchor(self):
        """Test extracting page_id and block_id from URL with anchor."""
        # Import function
        sys.path.insert(0, os.path.dirname(os.path.dirname(os.path.abspath(__file__))) + '/scripts')

        url = "https://www.notion.so/page-name-2be9aec621258070b6dae3699106662e#2bf9aec62125807f9b72ccc4daf3717c"

        # Extract IDs manually (same logic as script)
        url_clean = url.split('?')[0]
        block_id = None
        if '#' in url_clean:
            url_clean, anchor = url_clean.split('#', 1)
            if len(anchor) == 32:
                block_id = f"{anchor[:8]}-{anchor[8:12]}-{anchor[12:16]}-{anchor[16:20]}-{anchor[20:]}"

        page_id_raw = url_clean.rstrip('/').split('-')[-1]
        if len(page_id_raw) == 32:
            page_id = f"{page_id_raw[:8]}-{page_id_raw[8:12]}-{page_id_raw[12:16]}-{page_id_raw[16:20]}-{page_id_raw[20:]}"

        self.assertEqual(page_id, "2be9aec6-2125-8070-b6da-e3699106662e")
        self.assertEqual(block_id, "2bf9aec6-2125-807f-9b72-ccc4daf3717c")

    def test_extract_ids_from_url_without_anchor(self):
        """Test extracting page_id from URL without anchor."""
        url = "https://www.notion.so/page-name-2be9aec621258070b6dae3699106662e"

        url_clean = url.split('?')[0]
        block_id = None
        if '#' in url_clean:
            url_clean, anchor = url_clean.split('#', 1)
            block_id = anchor

        page_id_raw = url_clean.rstrip('/').split('-')[-1]
        page_id = f"{page_id_raw[:8]}-{page_id_raw[8:12]}-{page_id_raw[12:16]}-{page_id_raw[16:20]}-{page_id_raw[20:]}"

        self.assertEqual(page_id, "2be9aec6-2125-8070-b6da-e3699106662e")
        self.assertIsNone(block_id)

    def test_format_markdown(self):
        """Test markdown table formatting."""
        data = [
            ["Name", "Value"],
            ["Row1", "Val1"],
            ["Row2", "Val2"]
        ]

        lines = []
        header = data[0]
        lines.append("| " + " | ".join(header) + " |")
        lines.append("|" + "|".join(["---"] * len(header)) + "|")
        for row in data[1:]:
            lines.append("| " + " | ".join(row) + " |")
        result = "\n".join(lines)

        self.assertIn("| Name | Value |", result)
        self.assertIn("|---|---|", result)
        self.assertIn("| Row1 | Val1 |", result)

    def test_format_json(self):
        """Test JSON table formatting."""
        data = [
            ["Name", "Value"],
            ["Row1", "Val1"],
            ["Row2", "Val2"]
        ]

        header = data[0]
        rows = []
        for row in data[1:]:
            row_dict = {}
            for i, col in enumerate(header):
                row_dict[col] = row[i] if i < len(row) else ""
            rows.append(row_dict)

        self.assertEqual(len(rows), 2)
        self.assertEqual(rows[0]["Name"], "Row1")
        self.assertEqual(rows[0]["Value"], "Val1")


def run_tests():
    """Run all tests and return results."""
    loader = unittest.TestLoader()
    suite = unittest.TestSuite()

    # Add all test classes
    suite.addTests(loader.loadTestsFromTestCase(TestBlockIdExtraction))
    suite.addTests(loader.loadTestsFromTestCase(TestMermaidBlockCreation))
    suite.addTests(loader.loadTestsFromTestCase(TestAppendBlocksAfter))
    suite.addTests(loader.loadTestsFromTestCase(TestCalloutBlockCreation))
    suite.addTests(loader.loadTestsFromTestCase(TestDividerBlock))
    suite.addTests(loader.loadTestsFromTestCase(TestHeadingBlocks))
    suite.addTests(loader.loadTestsFromTestCase(TestTableExtraction))

    # Run with verbosity
    runner = unittest.TextTestRunner(verbosity=2)
    result = runner.run(suite)

    return result


if __name__ == "__main__":
    print("=" * 60)
    print("🧪 Notion Tasks Operations Skill - Test Suite")
    print("=" * 60)
    print()

    result = run_tests()

    print()
    print("=" * 60)
    if result.wasSuccessful():
        print("✅ ALL TESTS PASSED")
    else:
        print(f"❌ TESTS FAILED: {len(result.failures)} failures, {len(result.errors)} errors")
    print("=" * 60)
