---
name: obsidian
description: Primary knowledge base operations - vault navigation, .base tables, MCP access, entity system. Use when user says "obsidian", "найди в обсидиан", "create page in vault", "query base", "vault structure".
version: "1.0.0"
---

# Obsidian Knowledge Base

**Thesis:** Central skill for Obsidian vault operations — navigation, querying .base tables, MCP tools, and integration with Claude Code sessions.

## Vault Location & Access

```
VAULT_PATH = $PROJECT_ROOT/data_sources/obsidian/vault/
VAULT_NAME = vault
SAFE_CLI = ./data_sources/obsidian/ops/obsidian_cli_safe.sh
```

**Access Methods:**
1. **MCP Tools:** `mcp__obsidian__*` (via REST API plugin)
2. **Python ops:** `from data_sources.obsidian.ops import ...`
3. **Direct Read:** Files are markdown, readable with Read tool
4. **Obsidian CLI (safe):** always run through `SAFE_CLI` so commands are pinned to `VAULT_NAME`

## CLI Guardrails (Critical)

- Never run bare `obsidian <vault-scoped-command>` from repo root.
- Always use: `./data_sources/obsidian/ops/obsidian_cli_safe.sh <command> ...`
- If CLI starts opening repo root as vault:
  1. Quit Obsidian (Cmd+Q)
  2. Run `python3 data_sources/obsidian/ops/fix_obsidian_default_vault.py`
  3. Reopen Obsidian and retry via `SAFE_CLI`

## Vault Structure

```
vault/
├── Improvado/                          # 🏢 Company knowledge
│   ├── 01-Revenue/                    # Sales, Marketing, Customer Success
│   │   ├── 12-Marketing/             # Marketing analytics, campaigns
│   │   └── 15-Customer-Success/      # CSM docs, Customers/
│   ├── 05-Product/                    # Product features, entities, taxonomy
│   ├── 06-Engineering/                # Data pipelines, architecture
│   └── 2.3 Entity Types Tags/        # Entity taxonomy definitions
│
├── Miras Knowledge Platform/           # 🤖 Agent system documentation
│   ├── Miras Agent Integration/       # Hooks, workflows
│   ├── Miras Intelligence/            # Dashboards, analytics
│   └── Miras Platform Core/           # API, CLI, deployment
│
├── Data sources and vendors/           # 🔌 Integration docs
│   ├── Claude code/                   # Sessions, skills
│   │   ├── Claude Code Sessions base.base
│   │   └── Claude code Skills base.base
│   ├── Gong/
│   ├── Notion/
│   └── Obsidian/                      # Meta-docs about Obsidian itself
│
├── General Knowledge and Research/     # 📚 Cross-domain knowledge
│
├── Tags/                               # 🏷️ Tag folders (queryable)
│   ├── #customer/
│   ├── #overview/
│   └── #obsidian/
│
├── 2.3 Entity Types Tags/              # 📋 Entity taxonomy
│
└── Daniel Personal/ → symlink          # 🔗 Personal vault
```

## Related Skills

| Skill | Purpose | When to use |
|-------|---------|-------------|
| `/claude-task-from-obsidian` | Execute tasks with `#claude_task` tag | Autonomous task execution from vault |
| `/graph_find` | Parallel multi-algorithm vault search (5 methods) | Find existing context on any topic before creating/updating |
| `/graph_add` | Smart content addition (find → decide → create/update/refactor) | Add content to vault with dedup-aware placement |

## .base Files (Structured Tables)

`.base` files are YAML configs for MakeSpace plugin — define structured views over markdown files.

### Active .base Registry

| Base | Location | Purpose |
|------|----------|---------|
| **Claude Code Sessions base.base** | `Data sources and vendors/Claude code/` | All Claude sessions with metadata |
| **Claude code Skills base.base** | `Data sources and vendors/Claude code/` | Skills registry |
| **Customers.base** | `Improvado/01-Revenue/15-Customer-Success/Customers/` | Customer table with SFDC links |
| **Product Features.base** | `Improvado/05-Product/Product features taxonomy/` | Product feature taxonomy |
| **Entities by Department.base** | `Improvado/2.3 Entity Types Tags/` | Entity definitions by dept |
| **Data Sources.base** | `Data sources and vendors/` | Integration data sources |
| **Improvado.base** | `Improvado/` | Top-level company structure |
| **All Bases.base** | `Data sources and vendors/Obsidian/obsidian base/Base Registry/` | Meta-registry of all bases |

### .base File Format

```yaml
# Example: Customers.base
formulas:
  sfdc_link: if(sfdc_account_id, link("https://improvado.lightning.force.com/..." + sfdc_account_id + "/view", "SFDC"), "")

properties:
  company_name:
    displayName: Company
  agency_id:
    displayName: ID
  cluster:
    displayName: Cluster

views:
  - type: table
    name: All Customers
    filters:
      and:
        - file.ext == "md"
        - file.inFolder("Improvado/01-Revenue/15-Customer-Success/Customers")
        - file.hasTag("customer")
    order:
      - company_name
      - agency_id
      - cluster
    sort:
      - property: company_name
        direction: ASC
```

### Querying .base with Dataview

```dataview
TABLE company_name, agency_id, cluster
FROM "Improvado/01-Revenue/15-Customer-Success/Customers"
WHERE contains(tags, "#customer")
SORT company_name ASC
```

## MCP Tools

Obsidian REST API plugin provides MCP tools:

| Tool | Purpose |
|------|---------|
| `mcp__obsidian__create_vault_file` | Create new .md file in vault |
| `mcp__obsidian__patch_vault_file` | Append/modify existing file |
| `mcp__obsidian__show_file_in_obsidian` | Open file in Obsidian app |

### Usage Examples

```python
# Create file
mcp__obsidian__create_vault_file(
    filename="General Knowledge and Research/new_topic.md",
    content="# New Topic\n\nContent here..."
)

# Patch file (append)
mcp__obsidian__patch_vault_file(
    filename="path/to/file.md",
    target="## Section",
    targetType="heading",
    operation="append",
    content="- New bullet point"
)
```

## Python Operations

```python
import sys
sys.path.insert(0, '$PROJECT_ROOT')
from data_sources.obsidian.ops import (
    # Task operations
    find_pending_tasks,
    add_result,
    add_session_link,
    update_task_status,

    # Placement suggestions
    suggest_placement,
    build_vault_hierarchy,
    PlacementResult,
)

# Find where to put new content
result = suggest_placement("Gong call notes for GSK customer")
# Returns: PlacementResult(name="GSK", path="Improvado/01-Revenue/15-Customer-Success/Customers/gsk.com/", score=0.92)

# Get vault hierarchy for navigation
hierarchy = build_vault_hierarchy(depth=3)
print(hierarchy)
```

## Entity System (2.3 Entity Types Tags/)

Structured taxonomy for business concepts:

| Entity Type | Location | Example Tags |
|-------------|----------|--------------|
| **Customer** | `Tags/#customer/` | `#customer`, `#enterprise`, `#montana` |
| **Product Feature** | `05-Product/` | `#feature`, `#ai-agent`, `#integration` |
| **Data Source** | `Data sources and vendors/` | `#gong`, `#notion`, `#clickhouse` |
| **Overview** | `Tags/#overview/` | `#overview` for index pages |

### Frontmatter Properties

Customer pages have properties:
```yaml
---
company_name: GSK
agency_id: XXXX
dim_domain_id: gsk.com
cluster: Montana
sfdc_segment: Enterprise
sfdc_support_tier: Premium
sfdc_primary_csm_name: John Doe
tags:
  - customer
  - enterprise
  - montana
---
```

## Claude Sessions Integration

Sessions are tracked in `Data sources and vendors/Claude code/`:

```
Claude code/
├── Claude Code Sessions base.base     # Main sessions table
├── Claude code Skills base.base       # Skills registry
├── Claude code session/
│   ├── claude code base-views/        # Filtered views
│   │   ├── Sessions - Customers.base  # Customer-related sessions
│   │   ├── Sessions - Miras.base      # Miras-related sessions
│   │   └── Sessions - Obsidian.base   # Obsidian-related sessions
│   └── Sessions by Folder.base        # Sessions grouped by folder
```

### Linking Session to Vault

```python
from data_sources.obsidian.ops import add_session_link

# Add session reference to a task
add_session_link(
    file_path="/path/to/vault/file.md",
    line_number=5,
    session_id="abc123-def456"
)
# Adds: 📎 [[Claude Sessions/rc abc123-def456]]
```

## Quick Patterns

### ¶1 Find right place for content
```python
from data_sources.obsidian.ops import suggest_placement
result = suggest_placement("Marketing campaign analysis Q4")
print(f"Put it in: {result.path}")
```

### ¶2 Read vault hierarchy
```python
from data_sources.obsidian.ops import build_vault_hierarchy
print(build_vault_hierarchy(root="Improvado", depth=2))
```

### ¶3 Create customer page
```python
content = """---
company_name: New Customer
agency_id: 12345
cluster: Lisbon
tags:
  - customer
---

# New Customer

Customer notes here...
"""
mcp__obsidian__create_vault_file(
    filename="Improvado/01-Revenue/15-Customer-Success/Customers/newcustomer.com/New Customer.md",
    content=content
)
```

## Ground Truth

- **Vault path:** `data_sources/obsidian/vault/`
- **Python ops:** `data_sources/obsidian/ops/`
- **MCP config:** Obsidian REST API plugin
- **Plugins:** Dataview, MakeSpace (.base), Kanban, Templater

---
Created: 2026-01-25 | Session: auto
