#!/usr/bin/env python3
"""
Parse Claude Code session log and extract modified/read files.
Usage: python parse_session_files.py <session_id>
"""
import sys
import json
from pathlib import Path
from typing import Set, Tuple

def find_session_file(session_id: str) -> Path:
    """Find session log file"""
    # New format
    projects_dir = Path.home() / '.claude' / 'projects'
    if projects_dir.exists():
        files = list(projects_dir.glob(f"**/{session_id}.jsonl"))
        if files:
            return files[0]

    # Old format
    sessions_dir = Path.home() / '.codex' / 'sessions'
    if sessions_dir.exists():
        files = list(sessions_dir.glob(f"**/rollout-*{session_id}.jsonl"))
        if files:
            return files[0]

    raise FileNotFoundError(f"Session {session_id} not found")

def parse_files(log_file: Path) -> Tuple[Set[str], Set[str]]:
    """Extract modified and read files from session log"""
    modified = set()
    read = set()

    with open(log_file, 'r') as f:
        for line in f:
            if not line.strip():
                continue

            try:
                entry = json.loads(line)

                # New Claude Code format: entry.message.content[]
                if entry.get('type') == 'assistant':
                    message = entry.get('message', {})
                    content = message.get('content', [])

                    for item in content:
                        if isinstance(item, dict) and item.get('type') == 'tool_use':
                            name = item.get('name', '')
                            params = item.get('input', {})

                            if name == 'Read':
                                file_path = params.get('file_path', '')
                                if file_path:
                                    read.add(file_path)
                            elif name in ['Write', 'Edit']:
                                file_path = params.get('file_path', '')
                                if file_path:
                                    modified.add(file_path)

                # Old Claude Code format
                elif entry.get('type') == 'tool_use':
                    name = entry.get('name', '')
                    params = entry.get('input', {})

                    if name == 'Read':
                        file_path = params.get('file_path', '')
                        if file_path:
                            read.add(file_path)
                    elif name in ['Write', 'Edit']:
                        file_path = params.get('file_path', '')
                        if file_path:
                            modified.add(file_path)

                # Codex format
                elif entry.get('type') == 'response_item':
                    payload = entry.get('payload', {})
                    if payload.get('type') == 'function_call':
                        name = payload.get('name', '')
                        try:
                            params = json.loads(payload.get('arguments', '{}'))
                        except:
                            params = {}

                        if name in ['read_file', 'glob_search']:
                            file_path = params.get('file_path', '') or params.get('pattern', '')
                            if file_path:
                                read.add(file_path)
                        elif name in ['write_file', 'edit_file', 'create_file']:
                            file_path = params.get('file_path', '')
                            if file_path:
                                modified.add(file_path)

            except json.JSONDecodeError:
                continue

    return modified, read

def main():
    if len(sys.argv) < 2:
        print("Usage: python parse_session_files.py <session_id>")
        sys.exit(1)

    session_id = sys.argv[1]

    try:
        log_file = find_session_file(session_id)
        modified, read = parse_files(log_file)

        print("**Files Modified:**")
        for f in sorted(modified):
            print(f"- {f}")

        print("\n**Files Read:**")
        for f in sorted(read):
            print(f"- {f}")

        print(f"\n**Total:** {len(modified)} modified, {len(read)} read")

    except FileNotFoundError as e:
        print(f"Error: {e}", file=sys.stderr)
        sys.exit(1)
    except Exception as e:
        print(f"Error: {e}", file=sys.stderr)
        sys.exit(1)

if __name__ == '__main__':
    main()
