# Supabase Agent - Quick Start

## 🎯 What is this?

Universal skill for querying Supabase AI Agent database from **any repository**. Automatically loads credentials from chrome-extension-tcs/.env

## 🚀 Quick Usage (3 Steps)

### Step 1: Copy example to your project

```bash
cp $PROJECT_ROOT/.claude/skills/supabase-agent/example_simple.py ./
```

### Step 2: Run it

```bash
python example_simple.py
```

### Step 3: Customize for your needs

```python
import sys
from pathlib import Path

# Add skill to path
SKILL_PATH = Path('$PROJECT_ROOT/.claude/skills/supabase-agent')
sys.path.insert(0, str(SKILL_PATH))

from supabase_client import get_supabase_client

# Your code here
client = get_supabase_client()
results = client.execute_query("SELECT * FROM chats LIMIT 10")
client.close()
```

## 📚 Main Tables

| Table | Purpose |
|-------|---------|
| `chats` | Chat sessions with metadata |
| `messages` | Chat messages (content in `parts` array) |
| `chats_extended_view` | Chats with agency_id and user_email |
| `agency_repo` | Agency repository mappings |

## 🔗 Common Queries

### Find today's chats for agency
```python
chats = client.execute_query("""
    SELECT c.id, c.name, c.workspace_id, ce.agency_id
    FROM chats c
    JOIN chats_extended_view ce ON c.id = ce.id
    WHERE ce.agency_id = XXXX
    AND DATE(c.created_at) = CURRENT_DATE
""")
```

### Extract messages from chat
```python
messages = client.execute_query("""
    SELECT sender, parts, created_at
    FROM messages
    WHERE chat_id = %s
    ORDER BY created_at ASC
""", (chat_id,))
```

### Parse conversation
```python
for msg in messages:
    for part in msg.get('parts', []):
        if part.get('type') == 'text':
            print(part.get('text'))
```

## 📖 Full Documentation

- **Complete guide:** `README.md`
- **Usage examples:** `example_usage.md`
- **Test connection:** `python test_connection.py`

## 🔧 Troubleshooting

**Error: "Missing required environment variables"**
- Ensure `$PROJECT_ROOT/.env` exists
- Check credentials in Notion: https://www.notion.so/improvado-home/Supabase-env-1e99aec6212580feaae0c9f9c0c64634

**Error: "column does not exist"**
- Use `client.describe_table('table_name')` to check schema
- Remember: agency names are in ClickHouse, not Supabase

**Empty results**
- Check timezone (dates in PST)
- Use `CURRENT_DATE` for today's data
- Verify agency_id exists

## 💡 Key Features

✅ Works from ANY repository
✅ Auto-loads credentials from chrome-extension-tcs/.env
✅ Connection pooling for performance
✅ Dictionary results for easy access
✅ Built-in table introspection
✅ Context managers for safe connections

## 🎓 Learning Path

1. **Start here:** Run `test_connection.py` to verify setup
2. **Basic queries:** Try `example_simple.py`
3. **Advanced patterns:** Read `example_usage.md`
4. **Deep dive:** Study `README.md` sections 1.0-7.0

---

**Need help?** Check full documentation in `README.md`
