# Supabase Agent Skill - Created Successfully! ✅

## 📦 What Was Created

Universal skill for connecting to Supabase AI Agent database from ANY repository.

**Location:** `.claude/skills/supabase-agent/`

## 📁 Skill Structure

```
.claude/skills/supabase-agent/
├── README.md              # Complete documentation (19KB)
├── QUICKSTART.md          # Quick start guide
├── skill.json             # Skill metadata
├── supabase_client.py     # Universal client (works from any repo)
├── example_simple.py      # Simple usage example
├── example_usage.md       # Detailed usage patterns
└── test_connection.py     # Connection test script
```

## ✨ Key Features

1. **Universal Access** - Works from ANY repository
2. **Auto-loads Credentials** - Always uses chrome-extension-tcs/.env
3. **Connection Pooling** - Performance optimized
4. **Dictionary Results** - Easy data access
5. **Table Introspection** - Built-in schema exploration
6. **Safe Connections** - Context managers prevent leaks

## 🚀 Quick Usage

### From Any Repository

```python
import sys
from pathlib import Path

# Add skill to path
SKILL_PATH = Path('$PROJECT_ROOT/.claude/skills/supabase-agent')
sys.path.insert(0, str(SKILL_PATH))

from supabase_client import get_supabase_client

# Query Supabase
client = get_supabase_client()
results = client.execute_query("SELECT * FROM chats LIMIT 10")
client.close()
```

### Copy Example to Your Project

```bash
# Copy pre-made example
cp $PROJECT_ROOT/.claude/skills/supabase-agent/example_simple.py ./

# Run it
python example_simple.py
```

## 📊 Main Tables

| Table | Purpose | Key Fields |
|-------|---------|------------|
| `chats` | Chat sessions | id, name, workspace_id, created_at |
| `messages` | Chat messages | chat_id, sender, parts (content!), created_at |
| `chats_extended_view` | Chats with agency info | + agency_id, user_email |
| `agency_repo` | Repository mappings | agency_id, repo_folder_url |

**CRITICAL:** Message content is in `parts` array, NOT `message_text` field!

## 🔧 Common Operations

### Find Today's Chats by Agency
```python
chats = client.execute_query("""
    SELECT c.id, c.name, c.workspace_id
    FROM chats c
    JOIN chats_extended_view ce ON c.id = ce.id
    WHERE ce.agency_id = XXXX
    AND DATE(c.created_at) = CURRENT_DATE
""")
```

### Extract Conversation
```python
messages = client.execute_query("""
    SELECT sender, parts, created_at
    FROM messages
    WHERE chat_id = %s
    ORDER BY created_at ASC
""", (chat_id,))

for msg in messages:
    for part in msg.get('parts', []):
        if part.get('type') == 'text':
            print(part.get('text'))
```

### Generate Chat URL
```python
url = f"https://report.improvado.io/experimental/agent/assistant/{chat_id}/?agency_id={agency_id}&workspace_id={workspace_id}"
```

## ✅ Tests Passed

1. ✅ Connection test from chrome-extension-tcs repo
2. ✅ Connection test from /tmp directory (different repo)
3. ✅ Query execution and data retrieval
4. ✅ Table introspection
5. ✅ Auto-loading credentials from chrome-extension-tcs/.env

**Test Results:**
- Total tables: 56
- Total chats: 13,933
- Unique users: 560
- Today's chats: 46

## 📖 Documentation

- **Quick Start:** `QUICKSTART.md` - Get started in 3 steps
- **Complete Guide:** `README.md` - Full documentation with sections:
  - 1.0 Database Connection
  - 2.0 Database Structure
  - 3.0 Common Queries
  - 4.0 Chat URL Generation
  - 5.0 Using From Other Repositories
  - 6.0 Complete Example Scripts
  - 7.0 Related Documentation
- **Usage Patterns:** `example_usage.md` - Real-world use cases
- **Test Script:** `test_connection.py` - Verify setup

## 🎯 Usage Scenarios

### Scenario 1: Daily Activity Report
```python
# Copy example to your project
cp .claude/skills/supabase-agent/example_simple.py ./daily_report.py

# Customize queries
# Run daily: python daily_report.py
```

### Scenario 2: User Support
```python
# Find user's recent chats
chats = client.execute_query("""
    SELECT * FROM chats c
    JOIN chats_extended_view ce ON c.id = ce.id
    WHERE ce.user_email = 'user@example.com'
    ORDER BY c.created_at DESC
    LIMIT 10
""")
```

### Scenario 3: Analytics Dashboard
```python
# Query usage metrics
stats = client.execute_query("""
    SELECT agency_id, COUNT(*) as chat_count
    FROM chats_extended_view
    WHERE created_at >= CURRENT_DATE - INTERVAL '7 days'
    GROUP BY agency_id
    ORDER BY chat_count DESC
""")
```

## 🔑 Credentials Setup

**Already configured!** Credentials are in:
```
$PROJECT_ROOT/.env
```

Variables used:
- `EA_AGENT_POSTGRES_HOST`
- `EA_AGENT_POSTGRES_POOL_PORT`
- `EA_AGENT_POSTGRES_DATABASE`
- `EA_AGENT_POSTGRES_USER`
- `EA_AGENT_POSTGRES_PASSWORD`

**Backup location:** Notion - https://www.notion.so/improvado-home/Supabase-env-1e99aec6212580feaae0c9f9c0c64634

## 🎓 Next Steps

1. **Test the skill:**
   ```bash
   python .claude/skills/supabase-agent/test_connection.py
   ```

2. **Try simple example:**
   ```bash
   python .claude/skills/supabase-agent/example_simple.py
   ```

3. **Copy to your project:**
   ```bash
   cp .claude/skills/supabase-agent/example_simple.py ~/your-project/
   cd ~/your-project
   python example_simple.py
   ```

4. **Read documentation:**
   - Quick start: `.claude/skills/supabase-agent/QUICKSTART.md`
   - Full guide: `.claude/skills/supabase-agent/README.md`

## 💡 Pro Tips

1. **Always use connection pooling:**
   ```python
   client = get_supabase_client(use_pool=True)  # Default
   ```

2. **Always close connections:**
   ```python
   try:
       # queries
   finally:
       client.close()
   ```

3. **Content is in parts array:**
   ```python
   # ❌ WRONG
   text = msg['message_text']

   # ✅ CORRECT
   for part in msg['parts']:
       if part['type'] == 'text':
           text = part['text']
   ```

4. **Use views for joins:**
   ```python
   # Use chats_extended_view instead of joining manually
   FROM chats_extended_view  # Has agency_id, user_email
   ```

## 🎉 Success!

Supabase Agent skill is ready to use from any repository!

**Quick verification:**
```bash
cd /tmp
python3 -c "
import sys; sys.path.insert(0, '$PROJECT_ROOT/.claude/skills/supabase-agent')
from supabase_client import get_supabase_client
client = get_supabase_client()
print('✅ Works!')
client.close()
"
```
