#!/usr/bin/env python3
"""
Simple example of using supabase-agent skill from any repository.

This script can be copied to any project and will automatically
load credentials from chrome-extension-tcs/.env

Usage:
    # Copy this file to your project
    cp /path/to/.claude/skills/supabase-agent/example_simple.py ./

    # Run from any directory
    python example_simple.py
"""

import sys
from pathlib import Path

# Add supabase-agent skill to path
SKILL_PATH = Path('$PROJECT_ROOT/.claude/skills/supabase-agent')
sys.path.insert(0, str(SKILL_PATH))

from supabase_client import get_supabase_client


def main():
    """Simple example: find today's AI Agent chats"""

    print("🔍 Finding today's AI Agent chats...")
    print("=" * 80)

    # Create client (automatically loads credentials from chrome-extension-tcs/.env)
    client = get_supabase_client()

    try:
        # Query today's chats
        chats = client.execute_query("""
            SELECT
                c.id,
                c.name,
                c.created_at,
                c.workspace_id,
                ce.agency_id,
                ce.user_email
            FROM chats c
            JOIN chats_extended_view ce ON c.id = ce.id
            WHERE DATE(c.created_at) = CURRENT_DATE
            ORDER BY c.created_at DESC
            LIMIT 10
        """)

        print(f"\n✅ Found {len(chats)} chat(s) today:\n")

        for i, chat in enumerate(chats, 1):
            # Generate chat URL
            url = f"https://report.improvado.io/experimental/agent/assistant/{chat['id']}/?agency_id={chat['agency_id']}&workspace_id={chat['workspace_id']}"

            print(f"{i}. {chat['name']}")
            print(f"   Time: {chat['created_at']}")
            print(f"   User: {chat['user_email']}")
            print(f"   Agency: {chat['agency_id']}")
            print(f"   URL: {url}")
            print()

    finally:
        client.close()


if __name__ == "__main__":
    main()
