# Supabase Agent - Example Usage

## Quick Start Examples

### 1. Test Connection

```bash
# Run connection test
python .claude/skills/supabase-agent/test_connection.py
```

Expected output:
```
🔌 Testing Supabase AI Agent connection...

📋 Available tables:
  - agencies
  - agency_repo
  - chats
  - messages
  ...

💬 Chat statistics:
  Total chats: 13,933
  Unique users: 560
  ...

✅ Connection test successful!
```

---

### 2. Find Today's Chats for Agency

```python
from data_sources.supabase.ea_agent_client import EAAgentClient

client = EAAgentClient(use_pool=True)

# Get today's chats for ExampleClient (agency_id = XXXX)
chats = client.execute_query("""
    SELECT
        c.id,
        c.name,
        c.created_at,
        c.workspace_id,
        ce.user_email
    FROM chats c
    JOIN chats_extended_view ce ON c.id = ce.id
    WHERE ce.agency_id = XXXX
    AND DATE(c.created_at) = CURRENT_DATE
    ORDER BY c.created_at DESC
""")

for chat in chats:
    url = f"https://report.improvado.io/experimental/agent/assistant/{chat['id']}/?agency_id=XXXX&workspace_id={chat['workspace_id']}"
    print(f"{chat['name']}: {url}")

client.close()
```

---

### 3. Extract Conversation from Chat

```python
from data_sources.supabase.ea_agent_client import EAAgentClient

client = EAAgentClient(use_pool=True)

chat_id = "your-chat-uuid-here"

# Get messages
messages = client.execute_query("""
    SELECT sender, parts, created_at
    FROM messages
    WHERE chat_id = %s
    ORDER BY created_at ASC
""", (chat_id,))

# Parse conversation
for msg in messages:
    print(f"\n{msg['sender'].upper()}:")
    for part in msg.get('parts', []):
        if part.get('type') == 'text':
            print(part.get('text'))

client.close()
```

---

### 4. Find Most Active Users (Last 7 Days)

```python
from data_sources.supabase.ea_agent_client import EAAgentClient

client = EAAgentClient(use_pool=True)

users = client.execute_query("""
    SELECT
        ce.user_email,
        ce.agency_id,
        COUNT(*) as chat_count
    FROM chats c
    JOIN chats_extended_view ce ON c.id = ce.id
    WHERE c.created_at >= CURRENT_DATE - INTERVAL '7 days'
    GROUP BY ce.user_email, ce.agency_id
    ORDER BY chat_count DESC
    LIMIT 10
""")

for user in users:
    print(f"{user['user_email']} (Agency {user['agency_id']}): {user['chat_count']} chats")

client.close()
```

---

### 5. Analyze Tool Usage

```python
from data_sources.supabase.ea_agent_client import EAAgentClient
import json

client = EAAgentClient(use_pool=True)

# Get messages with tool invocations from last 24 hours
messages = client.execute_query("""
    SELECT parts
    FROM messages
    WHERE created_at >= NOW() - INTERVAL '24 hours'
""")

tool_counts = {}
for msg in messages:
    for part in msg.get('parts', []):
        if part.get('type') == 'tool-invocation':
            tool_name = part.get('toolInvocation', {}).get('toolName')
            if tool_name:
                tool_counts[tool_name] = tool_counts.get(tool_name, 0) + 1

print("\n🔧 Tool Usage (Last 24 hours):")
for tool, count in sorted(tool_counts.items(), key=lambda x: x[1], reverse=True):
    print(f"  {tool}: {count} invocations")

client.close()
```

---

## Common Use Cases

### Use Case 1: Daily Activity Report
**Scenario:** Generate daily report of AI Agent activity for specific agency

**Steps:**
1. Query today's chats using `chats_extended_view`
2. Count messages per chat
3. Extract tool invocations
4. Generate summary with chat URLs

---

### Use Case 2: User Support
**Scenario:** Help user find their recent conversation

**Steps:**
1. Get user_email from support ticket
2. Query chats by user_email via `chats_extended_view`
3. Generate chat URLs with correct parameters
4. Provide URLs to user

---

### Use Case 3: Analytics Dashboard
**Scenario:** Build analytics on AI Agent usage patterns

**Steps:**
1. Query `agency_usage_view` for aggregated stats
2. Join with `chats_extended_view` for agency context
3. Analyze tool invocation patterns from `messages.parts`
4. Generate visualizations

---

## Tips & Best Practices

1. **Always use connection pooling** for production scripts:
   ```python
   client = EAAgentClient(use_pool=True)
   ```

2. **Close connections** when done:
   ```python
   try:
       # Your queries
   finally:
       client.close()
   ```

3. **Parse message parts correctly** - content is in `parts` array, NOT `message_text`

4. **Use views for joins** - `chats_extended_view` has pre-joined agency_id and user_email

5. **Generate correct URLs** - always include both `agency_id` and `workspace_id` parameters

---

## Troubleshooting

### Error: "column does not exist"
- Check table schema with `client.describe_table('table_name')`
- Many tables don't have agency_name - join with ClickHouse instead

### Error: "connection refused"
- Verify `.env` credentials are set
- Check network connection
- Ensure using correct port (5432 for direct, 6543 for pooling)

### Empty results
- Check timezone - dates are in PST
- Verify agency_id exists in data
- Use `CURRENT_DATE` for today's data

---

## Related Resources

- **Full Documentation:** `.claude/skills/supabase-agent/README.md`
- **Complete Query Guide:** `00_SUPABASE_AI_AGENT_QUERY_GUIDE.md`
- **Client Code:** `data_sources/supabase/ea_agent_client.py`
- **Search Guide:** `data_sources/supabase/01_search_ai_agent_chats_guide.md`
