#!/usr/bin/env python3
"""
Universal Supabase AI Agent Client

This client can be used from any repository by loading credentials from:
1. Already set environment variables
2. Local .env file
3. Shared .env file specified in SUPABASE_AGENT_ENV_PATH

Usage:
    from supabase_client import get_supabase_client

    client = get_supabase_client()
    results = client.execute_query("SELECT * FROM chats LIMIT 10")
    client.close()
"""

import os
import sys
import psycopg2
import psycopg2.pool
from typing import List, Dict, Any, Optional
from contextlib import contextmanager
from pathlib import Path


def load_env_from_path(env_path: Optional[str] = None):
    """
    Load environment variables from .env file.

    Priority:
    1. If env_path provided, use it
    2. ALWAYS use chrome-extension-tcs/.env (main credentials source)
    3. Check local .env (for overrides)
    """
    from dotenv import load_dotenv

    # Always load chrome-extension-tcs .env first (base credentials)
    default_env = Path('$PROJECT_ROOT/.env')
    if default_env.exists():
        load_dotenv(default_env)
        print(f"📁 Loaded base credentials from: {default_env}")

    # Then load custom path if provided (overrides)
    if env_path and Path(env_path).exists():
        load_dotenv(env_path, override=True)
        print(f"📁 Loaded overrides from: {env_path}")
        return env_path

    # Or load local .env if exists (overrides)
    if Path('.env').exists():
        load_dotenv('.env', override=True)
        print(f"📁 Loaded local overrides from: .env")
        return '.env'

    return str(default_env)


class SupabaseAgentClient:
    """PostgreSQL client for Supabase AI Agent database"""

    def __init__(self, use_pool: bool = True, env_path: Optional[str] = None):
        """
        Initialize the Supabase AI Agent PostgreSQL client

        Args:
            use_pool: Whether to use connection pooling (recommended for production)
            env_path: Path to .env file (optional, auto-detected if not provided)
        """
        self.use_pool = use_pool

        # Load environment variables
        loaded_env = load_env_from_path(env_path)
        if loaded_env:
            print(f"📁 Loaded credentials from: {loaded_env}")

        # Get connection parameters from environment
        self.host = os.getenv('EA_AGENT_POSTGRES_HOST')
        self.port = int(os.getenv('EA_AGENT_POSTGRES_POOL_PORT' if use_pool else 'EA_AGENT_POSTGRES_PORT', 5432))
        self.database = os.getenv('EA_AGENT_POSTGRES_DATABASE', 'postgres')
        self.user = os.getenv('EA_AGENT_POSTGRES_USER', 'postgres')
        self.password = os.getenv('EA_AGENT_POSTGRES_PASSWORD')

        if not all([self.host, self.password]):
            raise ValueError(
                "Missing required environment variables for Supabase connection.\n"
                "Set SUPABASE_AGENT_ENV_PATH or add .env file with EA_AGENT_POSTGRES_* variables.\n"
                "See .claude/skills/supabase-agent/README.md for setup instructions."
            )

        # Initialize connection pool
        if use_pool:
            self.pool = psycopg2.pool.SimpleConnectionPool(
                1, 20,  # min and max connections
                host=self.host,
                port=self.port,
                database=self.database,
                user=self.user,
                password=self.password,
                connect_timeout=10
            )
        else:
            self.pool = None

    @contextmanager
    def get_connection(self):
        """Get a database connection from the pool or create a new one"""
        conn = None
        try:
            if self.pool:
                conn = self.pool.getconn()
            else:
                conn = psycopg2.connect(
                    host=self.host,
                    port=self.port,
                    database=self.database,
                    user=self.user,
                    password=self.password,
                    connect_timeout=10
                )
            yield conn
        finally:
            if conn:
                if self.pool:
                    self.pool.putconn(conn)
                else:
                    conn.close()

    def execute_query(self, query: str, params: Optional[tuple] = None) -> List[Dict[str, Any]]:
        """
        Execute a SELECT query and return results as list of dictionaries

        Args:
            query: SQL query to execute
            params: Query parameters (optional)

        Returns:
            List of dictionaries with query results
        """
        with self.get_connection() as conn:
            with conn.cursor() as cursor:
                cursor.execute(query, params)

                # Get column names
                columns = [desc[0] for desc in cursor.description]

                # Fetch all results and convert to dictionaries
                results = []
                for row in cursor.fetchall():
                    results.append(dict(zip(columns, row)))

                return results

    def execute_command(self, command: str, params: Optional[tuple] = None) -> int:
        """
        Execute an INSERT, UPDATE, or DELETE command

        Args:
            command: SQL command to execute
            params: Command parameters (optional)

        Returns:
            Number of affected rows
        """
        with self.get_connection() as conn:
            with conn.cursor() as cursor:
                cursor.execute(command, params)
                conn.commit()
                return cursor.rowcount

    def list_tables(self, schema: str = 'public') -> List[str]:
        """List all tables in the specified schema"""
        query = """
        SELECT table_name
        FROM information_schema.tables
        WHERE table_schema = %s
        ORDER BY table_name
        """
        results = self.execute_query(query, (schema,))
        return [row['table_name'] for row in results]

    def describe_table(self, table_name: str, schema: str = 'public') -> List[Dict[str, Any]]:
        """Get table schema information"""
        query = """
        SELECT
            column_name,
            data_type,
            is_nullable,
            column_default,
            character_maximum_length
        FROM information_schema.columns
        WHERE table_schema = %s AND table_name = %s
        ORDER BY ordinal_position
        """
        return self.execute_query(query, (schema, table_name))

    def close(self):
        """Close the connection pool"""
        if self.pool:
            self.pool.closeall()


def get_supabase_client(use_pool: bool = True, env_path: Optional[str] = None) -> SupabaseAgentClient:
    """
    Convenience function to create Supabase AI Agent client.

    Args:
        use_pool: Whether to use connection pooling (default: True)
        env_path: Path to .env file (optional, auto-detected if not provided)

    Returns:
        SupabaseAgentClient instance

    Example:
        client = get_supabase_client()
        results = client.execute_query("SELECT * FROM chats LIMIT 10")
        client.close()
    """
    return SupabaseAgentClient(use_pool=use_pool, env_path=env_path)


def main():
    """Test the Supabase AI Agent connection"""
    try:
        # Create client
        print("🔌 Connecting to Supabase AI Agent database...")
        client = get_supabase_client(use_pool=True)

        # Test connection by listing tables
        print("\n📋 Available tables:")
        tables = client.list_tables()
        for table in tables[:10]:
            print(f"  - {table}")
        if len(tables) > 10:
            print(f"  ... and {len(tables) - 10} more")

        # Show database info
        print(f"\n📊 Database Info:")
        print(f"  Host: {client.host}")
        print(f"  Port: {client.port}")
        print(f"  Database: {client.database}")
        print(f"  User: {client.user}")
        print(f"  Using Pool: {client.use_pool}")

        # Test query
        print(f"\n💬 Quick stats:")
        result = client.execute_query("""
            SELECT
                COUNT(*) as total_chats,
                COUNT(DISTINCT user_id) as unique_users
            FROM chats
        """)
        stats = result[0]
        print(f"  Total chats: {stats['total_chats']:,}")
        print(f"  Unique users: {stats['unique_users']:,}")

        # Close connection
        client.close()
        print("\n✅ Connection test successful!")

    except Exception as e:
        print(f"\n❌ Error: {e}")
        return 1

    return 0


if __name__ == "__main__":
    sys.exit(main())
