#!/usr/bin/env python3
"""
Quick test script for Supabase AI Agent connection.
Usage: python test_connection.py
"""

import sys
import os

# Add project root to path
sys.path.insert(0, '$PROJECT_ROOT')

from data_sources.supabase.ea_agent_client import EAAgentClient

def test_connection():
    """Test Supabase AI Agent connection and display basic info."""

    print("🔌 Testing Supabase AI Agent connection...\n")

    try:
        # Create client
        client = EAAgentClient(use_pool=True)

        # Test 1: List tables
        print("📋 Available tables:")
        tables = client.list_tables()
        for table in tables[:10]:  # Show first 10
            print(f"  - {table}")
        if len(tables) > 10:
            print(f"  ... and {len(tables) - 10} more")

        # Test 2: Count chats
        print("\n💬 Chat statistics:")
        result = client.execute_query("""
            SELECT
                COUNT(*) as total_chats,
                COUNT(DISTINCT user_id) as unique_users,
                MIN(created_at) as first_chat,
                MAX(created_at) as last_chat
            FROM chats
        """)
        stats = result[0]
        print(f"  Total chats: {stats['total_chats']:,}")
        print(f"  Unique users: {stats['unique_users']:,}")
        print(f"  First chat: {stats['first_chat']}")
        print(f"  Last chat: {stats['last_chat']}")

        # Test 3: Today's activity
        print("\n📅 Today's activity:")
        today_result = client.execute_query("""
            SELECT COUNT(*) as today_chats
            FROM chats
            WHERE DATE(created_at) = CURRENT_DATE
        """)
        print(f"  Chats today: {today_result[0]['today_chats']}")

        # Test 4: Top agencies by chat count
        print("\n🏢 Top 5 agencies by chat count (last 7 days):")
        agencies = client.execute_query("""
            SELECT
                ce.agency_id,
                COUNT(*) as chat_count
            FROM chats c
            JOIN chats_extended_view ce ON c.id = ce.id
            WHERE c.created_at >= CURRENT_DATE - INTERVAL '7 days'
            AND ce.agency_id IS NOT NULL
            GROUP BY ce.agency_id
            ORDER BY chat_count DESC
            LIMIT 5
        """)

        for i, agency in enumerate(agencies, 1):
            print(f"  {i}. Agency ID {agency['agency_id']} - {agency['chat_count']} chats")

        # Close connection
        client.close()

        print("\n✅ Connection test successful!")
        return 0

    except Exception as e:
        print(f"\n❌ Connection test failed: {e}")
        import traceback
        traceback.print_exc()
        return 1

if __name__ == "__main__":
    sys.exit(test_connection())
