#!/usr/bin/env python3
"""
Upload Claude Code Session to Google Drive

Created by: Claude Code
Session ID: [current_session]
Date: 2025-11-11
Purpose: Skill script to upload current Claude Code session to Google Drive with automatic authentication
"""

import os
import sys
from pathlib import Path
from datetime import datetime
import json

# Add project root to path
project_root = Path(__file__).parent.parent.parent.parent
sys.path.insert(0, str(project_root))

from data_sources.google_drive.drive_client import GoogleDriveClient


def get_current_session_id() -> str:
    """
    Get current Claude Code session ID.

    Tries multiple methods:
    1. CLAUDE_SESSION_ID environment variable
    2. Most recent session file in current project

    Returns:
        Session ID (UUID format)
    """
    # Method 1: Environment variable
    session_id = os.environ.get('CLAUDE_SESSION_ID')
    if session_id:
        print(f"📋 Session ID from environment: {session_id}")
        return session_id

    # Method 2: Find most recent session in current project
    claude_dir = Path.home() / '.claude' / 'projects'

    # Encode current working directory to match Claude's folder naming
    cwd = os.getcwd()
    encoded_path = cwd.replace('/', '-')
    project_dir = claude_dir / encoded_path

    if not project_dir.exists():
        raise Exception(
            f"Project directory not found: {project_dir}\n"
            f"Expected for cwd: {cwd}"
        )

    # Find most recent .jsonl file
    session_files = list(project_dir.glob('*.jsonl'))
    if not session_files:
        raise Exception(f"No session files found in {project_dir}")

    # Sort by modification time, get most recent
    session_files.sort(key=lambda f: f.stat().st_mtime, reverse=True)
    latest_session = session_files[0]
    session_id = latest_session.stem  # Filename without extension

    print(f"📋 Session ID from latest file: {session_id}")
    return session_id


def find_session_file(session_id: str) -> Path:
    """
    Find session .jsonl file in ~/.claude/projects/

    Args:
        session_id: UUID of session

    Returns:
        Path to session file

    Raises:
        FileNotFoundError if session file not found
    """
    claude_dir = Path.home() / '.claude' / 'projects'

    # Search all project directories
    for project_dir in claude_dir.iterdir():
        if not project_dir.is_dir():
            continue

        session_file = project_dir / f"{session_id}.jsonl"
        if session_file.exists():
            print(f"📁 Found session file: {session_file}")
            return session_file

    raise FileNotFoundError(
        f"Session file not found for ID: {session_id}\n"
        f"Searched in: {claude_dir}"
    )


def get_session_metadata(session_file: Path) -> dict:
    """
    Extract metadata from first line of session file.

    Args:
        session_file: Path to .jsonl file

    Returns:
        Dictionary with session metadata
    """
    try:
        with open(session_file, 'r') as f:
            first_line = f.readline()
            metadata = json.loads(first_line)
            return {
                'session_id': metadata.get('sessionId', 'unknown'),
                'cwd': metadata.get('cwd', 'unknown'),
                'git_branch': metadata.get('gitBranch', 'unknown'),
                'version': metadata.get('version', 'unknown'),
                'timestamp': metadata.get('timestamp', 'unknown')
            }
    except Exception as e:
        print(f"⚠️  Could not read metadata: {e}")
        return {}


def upload_to_drive(
    session_file: Path,
    email: str = None,
    role: str = 'reader',
    folder_id: str = None
) -> dict:
    """
    Upload session to Google Drive with automatic authentication.

    Args:
        session_file: Path to session .jsonl file
        email: Optional email to share with
        role: Permission role (reader/writer/commenter)
        folder_id: Optional folder ID to upload into

    Returns:
        Dictionary with upload result
    """
    print("\n" + "="*60)
    print("🚀 Starting Google Drive upload...")
    print("="*60 + "\n")

    # Initialize client
    client = GoogleDriveClient()

    # Check if first time (token doesn't exist)
    token_path = Path(client.credentials_path) / 'token.pickle'
    if not token_path.exists():
        print("🔐 First time: Opening browser for Google Drive authentication...")
        print("Token will be saved for future automatic use.\n")

    # Authenticate (opens browser if needed, or reuses token)
    try:
        client.authenticate()
    except Exception as e:
        print(f"\n❌ Authentication failed: {e}")
        print("\nPlease ensure credentials.json exists at:")
        print(f"   {client.credentials_path}/credentials.json")
        raise

    # Upload file
    print(f"\n📤 Uploading session file...")
    print(f"   File: {session_file.name}")
    print(f"   Size: {session_file.stat().st_size / 1024:.1f} KB")

    try:
        if email:
            # Upload and share
            print(f"   Sharing with: {email} ({role})")
            result = client.upload_and_share(
                str(session_file),
                email,
                role=role,
                parent_folder_id=folder_id,
                notification_message=f"Claude Code session shared: {session_file.stem}"
            )

            if result:
                print(f"\n✅ Session uploaded and shared!")
                print(f"📧 Email notification sent to {email}")
            else:
                print(f"\n⚠️  Upload succeeded but sharing failed")

        else:
            # Upload only
            result = client.upload_file(
                str(session_file),
                parent_folder_id=folder_id
            )

            if result:
                print(f"\n✅ Session uploaded successfully!")

    except Exception as e:
        print(f"\n❌ Upload failed: {e}")
        raise

    return result


def main():
    """Main execution flow for skill."""
    try:
        print("\n" + "="*60)
        print("📤 Upload Session to Google Drive")
        print("="*60 + "\n")

        # Step 1: Get session ID
        print("Step 1: Getting current session ID...")
        session_id = get_current_session_id()

        # Step 2: Find session file
        print(f"\nStep 2: Finding session file...")
        session_file = find_session_file(session_id)

        # Step 3: Get metadata
        print(f"\nStep 3: Reading session metadata...")
        metadata = get_session_metadata(session_file)
        if metadata:
            print(f"   CWD: {metadata.get('cwd', 'unknown')}")
            print(f"   Git branch: {metadata.get('git_branch', 'unknown')}")
            print(f"   Claude version: {metadata.get('version', 'unknown')}")

        # Step 4: Check for email in arguments
        email = None
        if len(sys.argv) > 1:
            email = sys.argv[1]
            print(f"\n📧 Email for sharing: {email}")

        # Step 5: Upload to Drive
        print(f"\nStep 4: Uploading to Google Drive...")
        result = upload_to_drive(session_file, email=email)

        # Step 6: Display result
        print("\n" + "="*60)
        print("✅ SUCCESS!")
        print("="*60)
        print(f"\n🔗 Drive Link:")
        print(f"   {result['webViewLink']}")
        print(f"\n📋 File Details:")
        print(f"   Name: {result['name']}")
        print(f"   ID: {result['id']}")

        if email:
            print(f"\n👤 Shared with:")
            print(f"   Email: {result.get('shared_with', email)}")
            print(f"   Role: {result.get('role', 'reader')}")

        print("\n" + "="*60)

        return 0

    except Exception as e:
        print("\n" + "="*60)
        print("❌ ERROR")
        print("="*60)
        print(f"\n{str(e)}")
        print("\n" + "="*60)
        return 1


if __name__ == '__main__':
    sys.exit(main())
