# Colorización de Wide Events

Configuración de colores para logs legibles en desarrollo.

## Colores por Nivel

| Nivel | Color | ANSI Code | Uso |
|-------|-------|-----------|-----|
| ERROR | Rojo | `\x1b[31m` | Errores del sistema, excepciones |
| WARN | Amarillo | `\x1b[33m` | Warnings, 4xx, latencia alta |
| INFO | Verde | `\x1b[32m` | Requests exitosos normales |
| DEBUG | Gris | `\x1b[90m` | Debugging, detalles internos |

## Colores por Contexto

| Contexto | Color | ANSI Code | Campos |
|----------|-------|-----------|--------|
| Request | Azul | `\x1b[34m` | method, path, status, duration |
| User | Verde | `\x1b[32m` | id, plan, roles |
| Business | Magenta | `\x1b[35m` | action, entity_type, entity_id |
| Infra | Cyan | `\x1b[36m` | db, cache, external_calls |
| Error | Rojo | `\x1b[31m` | code, message, stack |
| Timestamp | Gris | `\x1b[90m` | timestamp, trace_id |

## Implementación Node.js con Pino

```typescript
import pino from 'pino';

const logger = pino({
  transport: {
    target: 'pino-pretty',
    options: {
      colorize: true,
      levelFirst: true,
      translateTime: 'HH:MM:ss',
      customColors: 'error:red,warn:yellow,info:green,debug:gray',
      messageFormat: '{request.method} {request.path} {request.status} {request.duration_ms}ms',
    },
  },
});
```

## Implementación Node.js con Picocolors

```typescript
import pc from 'picocolors';

const colors = {
  error: pc.red,
  warn: pc.yellow,
  info: pc.green,
  debug: pc.gray,
};

const contextColors = {
  request: pc.blue,
  user: pc.green,
  business: pc.magenta,
  infra: pc.cyan,
  error: pc.red,
  timestamp: pc.gray,
};

const formatWideEvent = (event: WideEvent): string => {
  const levelColor = colors[event.level] || pc.white;
  const level = levelColor(event.level.toUpperCase().padEnd(5));

  const timestamp = contextColors.timestamp(
    new Date(event.timestamp).toLocaleTimeString()
  );

  const request = contextColors.request(
    `${event.request.method} ${event.request.path} ${event.request.status} ${event.request.duration_ms}ms`
  );

  let output = `${timestamp} ${level} ${request}`;

  if (event.user) {
    output += `\n  ${contextColors.user(`user: ${event.user.id} (${event.user.plan})`)}`;
  }

  if (Object.keys(event.business).length > 0) {
    output += `\n  ${contextColors.business(`action: ${event.business.action}`)}`;
  }

  if (event.infra.db.queries_count > 0) {
    output += `\n  ${contextColors.infra(`db: ${event.infra.db.queries_count} queries, ${event.infra.db.total_time_ms}ms`)}`;
  }

  if (event.error) {
    output += `\n  ${contextColors.error(`error: ${event.error.code} - ${event.error.message}`)}`;
  }

  return output;
};
```

## Implementación Java con Logback

```xml
<!-- logback.xml -->
<configuration>
  <appender name="CONSOLE" class="ch.qos.logback.core.ConsoleAppender">
    <encoder class="ch.qos.logback.classic.encoder.PatternLayoutEncoder">
      <pattern>
        %d{HH:mm:ss} %highlight(%-5level) %cyan(%X{method}) %blue(%X{path}) %green(%X{status}) %yellow(%X{duration_ms}ms)%n
      </pattern>
    </encoder>
  </appender>

  <root level="INFO">
    <appender-ref ref="CONSOLE"/>
  </root>
</configuration>
```

## Implementación Java con ANSI Codes

```java
public class ColorFormatter {
    // ANSI escape codes
    private static final String RESET = "\u001B[0m";
    private static final String RED = "\u001B[31m";
    private static final String GREEN = "\u001B[32m";
    private static final String YELLOW = "\u001B[33m";
    private static final String BLUE = "\u001B[34m";
    private static final String MAGENTA = "\u001B[35m";
    private static final String CYAN = "\u001B[36m";
    private static final String GRAY = "\u001B[90m";

    public static String colorByLevel(String level, String text) {
        return switch (level.toLowerCase()) {
            case "error" -> RED + text + RESET;
            case "warn" -> YELLOW + text + RESET;
            case "info" -> GREEN + text + RESET;
            case "debug" -> GRAY + text + RESET;
            default -> text;
        };
    }

    public static String formatRequest(String method, String path, int status, long durationMs) {
        return BLUE + method + " " + path + RESET + " " +
               GREEN + status + RESET + " " +
               CYAN + durationMs + "ms" + RESET;
    }

    public static String formatError(String code, String message) {
        return RED + "ERROR: " + code + " - " + message + RESET;
    }
}
```

## Configuración por Ambiente

```typescript
const isDev = process.env.NODE_ENV === 'development';

const logWideEvent = (event: WideEvent) => {
  if (isDev) {
    // Formato coloreado y legible en desarrollo
    console.log(formatWideEvent(event));
  } else {
    // JSON puro en producción (para sistemas de log)
    console.log(JSON.stringify(event));
  }
};
```

## Output de Ejemplo

```
10:30:45 INFO  POST /api/orders 201 567ms
  user: user_789 (premium)
  action: create_order
  db: 6 queries, 180ms

10:30:46 WARN  GET /api/users/999 404 23ms
  error: NOT_FOUND - User not found

10:30:47 ERROR POST /api/payments 500 1234ms
  user: user_456 (free)
  action: process_payment
  db: 2 queries, 45ms
  error: PAYMENT_FAILED - Card declined
```
