# Implementación Wide Events

## Middleware TypeScript (Express/Fastify)

```typescript
import { AsyncLocalStorage } from 'async_hooks';

// Tipo del Wide Event
type WideEvent = {
  timestamp: string;
  level: 'debug' | 'info' | 'warn' | 'error';
  service: string;
  trace_id: string;
  deploy: {
    commit_hash: string;
    region: string;
    instance_id: string;
  };
  request: Record<string, unknown>;
  user: Record<string, unknown> | null;
  business: Record<string, unknown>;
  infra: {
    db: { queries_count: number; total_time_ms: number };
    cache: { hits: number; misses: number };
    external_calls: Array<Record<string, unknown>>;
  };
  error: Record<string, unknown> | null;
};

// Storage para acceso global al evento
const eventStorage = new AsyncLocalStorage<WideEvent>();

// Obtener evento actual
export const getWideEvent = (): WideEvent | undefined => eventStorage.getStore();

// Middleware
export const wideEventMiddleware = (serviceName: string) => {
  return (req: Request, res: Response, next: NextFunction) => {
    const startTime = Date.now();

    const wideEvent: WideEvent = {
      timestamp: new Date().toISOString(),
      level: 'info',
      service: serviceName,
      trace_id: req.headers['x-trace-id'] as string || crypto.randomUUID(),
      deploy: {
        commit_hash: process.env.GIT_COMMIT || process.env.COMMIT_HASH || 'unknown',
        region: process.env.AWS_REGION || process.env.REGION || 'local',
        instance_id: process.env.INSTANCE_ID || process.env.HOSTNAME || 'unknown',
      },
      request: {
        id: crypto.randomUUID(),
        method: req.method,
        path: req.path,
        route: req.route?.path,
        query: req.query,
        ip: req.ip,
      },
      user: null,
      business: {},
      infra: {
        db: { queries_count: 0, total_time_ms: 0 },
        cache: { hits: 0, misses: 0 },
        external_calls: [],
      },
      error: null,
    };

    // Capturar status y emitir al finalizar
    res.on('finish', () => {
      wideEvent.request.status = res.statusCode;
      wideEvent.request.duration_ms = Date.now() - startTime;

      // Ajustar level según resultado
      if (res.statusCode >= 500) wideEvent.level = 'error';
      else if (res.statusCode >= 400) wideEvent.level = 'warn';

      // Aplicar tail sampling y loggear
      if (shouldLog(wideEvent)) {
        console.log(JSON.stringify(wideEvent));
      }
    });

    // Ejecutar request con contexto
    eventStorage.run(wideEvent, () => next());
  };
};
```

## Propagación Cross-Service

Cuando llamás a otros servicios, propagá el `trace_id` para correlacionar logs.

```typescript
// Service A - Enviar trace_id en llamadas salientes
const callDownstreamService = async (url: string, data: unknown) => {
  const event = getWideEvent();
  const startTime = Date.now();

  const response = await fetch(url, {
    method: 'POST',
    headers: {
      'Content-Type': 'application/json',
      'x-trace-id': event?.trace_id || crypto.randomUUID(),
      'x-request-id': event?.request.id as string,
    },
    body: JSON.stringify(data),
  });

  // Registrar la llamada externa
  trackExternalCall(new URL(url).hostname, response.status, Date.now() - startTime);

  return response;
};

// Service B - Recibir y usar trace_id entrante
// (ya manejado en el middleware con req.headers['x-trace-id'])
```

**Headers estándar para propagación:**
- `x-trace-id` - ID único del trace (mismo en todos los servicios)
- `x-request-id` - ID del request original
- `x-parent-span-id` - (opcional) para tracing jerárquico

## Helpers de Enriquecimiento

```typescript
// Enriquecer con datos de usuario
export const enrichUser = (user: { id: string; plan?: string; roles?: string[] }) => {
  const event = getWideEvent();
  if (event) {
    event.user = { ...event.user, ...user };
  }
};

// Enriquecer con datos de negocio
export const enrichBusiness = (data: Record<string, unknown>) => {
  const event = getWideEvent();
  if (event) {
    event.business = { ...event.business, ...data };
  }
};

// Registrar query de DB
export const trackDbQuery = (durationMs: number) => {
  const event = getWideEvent();
  if (event) {
    event.infra.db.queries_count++;
    event.infra.db.total_time_ms += durationMs;
  }
};

// Registrar llamada externa
export const trackExternalCall = (service: string, status: number, durationMs: number) => {
  const event = getWideEvent();
  if (event) {
    event.infra.external_calls.push({ service, status, duration_ms: durationMs });
  }
};

// Registrar error
export const enrichError = (error: Error, code?: string) => {
  const event = getWideEvent();
  if (event) {
    event.error = {
      code: code || 'UNKNOWN_ERROR',
      message: error.message,
      type: error.name,
      stack: error.stack,
    };
    event.level = 'error';
  }
};
```

## Uso en Handlers

```typescript
app.post('/api/orders', async (req, res) => {
  try {
    // Enriquecer con usuario (ya autenticado por middleware)
    enrichUser({ id: req.user.id, plan: req.user.plan });

    // Lógica de negocio
    const order = await createOrder(req.body);

    // Enriquecer con datos de negocio
    enrichBusiness({
      action: 'create_order',
      entity_type: 'order',
      entity_id: order.id,
      order: {
        total: order.total,
        items_count: order.items.length,
      },
    });

    res.status(201).json(order);
  } catch (error) {
    enrichError(error, 'ORDER_CREATION_FAILED');
    res.status(500).json({ error: 'Failed to create order' });
  }
});
```

## Tail Sampling

```typescript
const shouldLog = (event: WideEvent): boolean => {
  // Siempre loggear errores
  if (event.level === 'error' || event.level === 'warn') return true;

  // Siempre loggear requests lentos (> 1s)
  if ((event.request.duration_ms as number) > 1000) return true;

  // Siempre loggear usuarios específicos (debug, VIP)
  if (event.user?.plan === 'enterprise') return true;

  // Samplear 10% del resto
  return Math.random() < 0.1;
};
```

## Queries SQL de Análisis

```sql
-- Requests más lentos por endpoint
SELECT
  request->>'path' as path,
  AVG((request->>'duration_ms')::int) as avg_duration,
  COUNT(*) as count
FROM logs
WHERE timestamp > NOW() - INTERVAL '1 hour'
GROUP BY request->>'path'
ORDER BY avg_duration DESC
LIMIT 10;

-- Errores por tipo
SELECT
  error->>'code' as error_code,
  COUNT(*) as count,
  array_agg(DISTINCT user->>'id') as affected_users
FROM logs
WHERE error IS NOT NULL
  AND timestamp > NOW() - INTERVAL '1 hour'
GROUP BY error->>'code'
ORDER BY count DESC;

-- Performance de servicios externos
SELECT
  call->>'service' as service,
  AVG((call->>'duration_ms')::int) as avg_duration,
  COUNT(*) FILTER (WHERE (call->>'status')::int >= 500) as errors
FROM logs,
  jsonb_array_elements(infra->'external_calls') as call
WHERE timestamp > NOW() - INTERVAL '1 hour'
GROUP BY call->>'service';
```
