---
name: shadergraph-editor
description: Create and edit ShaderGraph and RealityKit material networks in .usda files. Use when manually editing USD ASCII files to build, modify, or troubleshoot materials, shader nodes, and connections for RealityKit.
---

# ShaderGraph Editor (.usda)

## Overview

Use this skill to manually edit USD ASCII (.usda) files to create and modify RealityKit materials and ShaderGraph networks.

## Quick start workflow

1. Open the `.usda` file in a text editor.
2. Find or create a `def Material "MaterialName"` block in the correct scope.
3. Define a `def Shader "Surface"` prim and connect `outputs:surface` to the material output.
4. Add shader node prims (`def Shader`) for textures and math operations.
5. Connect node outputs to shader inputs with the `.connect` syntax.
6. Set constant values on inputs that are not connected.
7. Bind the material to geometry with `rel material:binding`.

## Core concepts

- **Material prim**: The root of a material definition in USD.
- **Surface shader**: A `Shader` prim that drives `outputs:surface`.
- **Shader nodes**: Additional `Shader` prims for textures, transforms, and math.
- **Connections**: `.connect` links between node outputs and shader inputs.
- **Material binding**: `rel material:binding` on a mesh prim.
- **info:id**: The shader node identifier used by USD and RealityKit.

## Implementation patterns

### Example 1: Basic red PBR material (UsdPreviewSurface)

```usda
def Material "RedMaterial"
{
    token outputs:surface.connect = <./Surface.outputs:surface>

    def Shader "Surface"
    {
        uniform token info:id = "UsdPreviewSurface"
        color3f inputs:diffuseColor = (1, 0, 0) # Red
        float inputs:roughness = 0.2
        float inputs:metallic = 0.0
        token outputs:surface
    }
}
```

### Example 2: Texture-mapped material

```usda
def Material "TexturedMaterial"
{
    token outputs:surface.connect = <./Surface.outputs:surface>

    def Shader "Surface"
    {
        uniform token info:id = "UsdPreviewSurface"
        color3f inputs:diffuseColor.connect = <../DiffuseTexture.outputs:rgb>
        token outputs:surface
    }

    def Shader "DiffuseTexture"
    {
        uniform token info:id = "UsdUVTexture"
        asset inputs:file = @textures/wood_albedo.png@
        float2 inputs:st.connect = <../PrimvarReader.outputs:result>
        float3 outputs:rgb
    }

    def Shader "PrimvarReader"
    {
        uniform token info:id = "UsdPrimvarReader_float2"
        string inputs:varname = "st" # Name of UV set on mesh
        float2 outputs:result
    }
}
```

### Example 3: RealityKit-specific nodes (conceptual)

```usda
def Material "UnlitMaterial"
{
    token outputs:surface.connect = <./UnlitSurface.outputs:surface>

    def Shader "UnlitSurface"
    {
        # Identifier may vary based on RealityKit version/export
        uniform token info:id = "ND_realitykit_unlit_surfaceshader"
        color3f inputs:color = (0, 1, 0)
        token outputs:surface
    }
}
```

## Pitfalls and checks

- Ensure `outputs:surface.connect` is present on the material.
- Verify `info:id` values match the expected node identifiers.
- Confirm all `.connect` paths point to valid outputs.
- Provide a `PrimvarReader` when using UV textures.
- Bind materials to geometry with `rel material:binding`.

## References

- [references/REFERENCE.md](references/REFERENCE.md) - ShaderGraph node and material reference guide.
