# VULNERABILITY SCANNING

## OVERVIEW
Vulnerability scanning identifies security weaknesses in code, dependencies, and infrastructure before attackers exploit them.

## SCANNING TYPES

### 1. SAST (Static Application Security Testing)

**Purpose**: Analyze source code for security vulnerabilities without executing code.

**Tools**:
- **PHP**: Psalm, PHPStan, Phan
- **Python**: Bandit, PyLint
- **JavaScript**: ESLint with security plugins, SonarQube
- **Java**: SpotBugs, FindSecurityBugs
- **Go**: Gosec, Staticcheck

**PHP Example (PHPStan)**:
```bash
# Install PHPStan
composer require --dev phpstan/phpstan

# Create phpstan.neon configuration
# phpstan.neon
parameters:
    level: 5
    checkMissingIterableValueType: true
    reportUnmatchedIgnoredErrors: false

# Run SAST scan
vendor/bin/phpstan analyse src --error-format=json > phpstan-report.json
```

**Python Example (Bandit)**:
```bash
# Install Bandit
pip install bandit[toml]

# Run SAST scan
bandit -r src/ -f json -o bandit-report.json

# Example output
# {"results": [{"severity": "HIGH", "code": "...", "filename": "..."}]}
```

---

### 2. DAST (Dynamic Application Security Testing)

**Purpose**: Test running application for security vulnerabilities by executing attacks.

**Tools**:
- **OWASP ZAP**: Free, open-source DAST tool
- **Burp Suite**: Commercial DAST tool
- **SQLMap**: Automated SQL injection testing
- **Nikto**: Web server scanner

**OWASP ZAP Example**:
```bash
# Run ZAP in daemon mode
zap.sh -daemon -host 0.0.0.0 -port 8080

# Spider application
zap-cli spider http://localhost:8080

# Active scan
zap-cli active-scan http://localhost:8080

# Generate report
zap-cli report -o zap-report.html -f html
```

**SQLMap Example**:
```bash
# Scan for SQL injection
sqlmap -u "http://localhost:8080/page?id=1" \
  --batch \
  --level=3 \
  --risk=2 \
  --output=sqlmap-report.txt
```

---

### 3. DEPENDENCY SCANNING

**Purpose**: Scan third-party libraries for known vulnerabilities.

**Tools**:
- **PHP**: Composer audit, Snyk
- **JavaScript**: npm audit, Snyk, Dependabot
- **Python**: pip-audit, Safety, Snyk
- **Java**: OWASP Dependency Check, Snyk

**PHP Example (Composer Audit)**:
```bash
# Run dependency audit
composer audit

# Output
# Found 1 security vulnerability advisory affecting 1 package.
```

**PHP Example (Snyk)**:
```bash
# Install Snyk
npm install -g snyk

# Scan dependencies
snyk test --file=composer.json

# Output
# Vulnerability found in vendor/package (CVE-2023-1234)
```

**JavaScript Example (npm audit)**:
```bash
# Run dependency audit
npm audit

# Fix vulnerabilities automatically
npm audit fix

# Output
# found 3 vulnerabilities (2 moderate, 1 high)
```

**Python Example (pip-audit)**:
```bash
# Install pip-audit
pip install pip-audit

# Scan dependencies
pip-audit

# Output
# Found 2 known vulnerabilities in 1 package
```

---

### 4. CONTAINER IMAGE SCANNING

**Purpose**: Scan Docker/OCI images for vulnerabilities.

**Tools**:
- **Trivy**: Fast, comprehensive vulnerability scanner
- **Clair**: Open-source vulnerability scanner
- **Docker Bench**: Docker security configuration checks
- **Grype**: Container vulnerability scanner

**Trivy Example**:
```bash
# Install Trivy
curl -sfL https://raw.githubusercontent.com/aquasecurity/trivy/main/contrib/install.sh | sh -s -- -b /usr/local/bin

# Scan image for vulnerabilities
trivy image myapp:latest

# Scan with severity filter
trivy image --severity HIGH,CRITICAL myapp:latest

# Scan for configuration issues
trivy config ./Dockerfile

# Output
# 2023-12-29T10:00:00Z    INFO    Number of language-specific files: 1
# myapp:latest (alpine 3.18.0)
# ==========================================
# Total: 2 (UNKNOWN: 0, LOW: 0, MEDIUM: 0, HIGH: 2, CRITICAL: 0)
```

**Docker Bench Example**:
```bash
# Run Docker Bench
docker run --rm --net host --pid host --cap-add audit_control \
  -v /var/lib:/var/lib \
  -v /var/run/docker.sock:/var/run/docker.sock \
  docker-bench-security

# Output
# [INFO] 1 - Host Configuration
# [INFO] 1.1  - Ensure a separate partition for containers has been created
# [PASS]
```

---

### 5. INFRASTRUCTURE SCANNING

**Purpose**: Scan servers, networks, and cloud infrastructure for vulnerabilities.

**Tools**:
- **Nmap**: Network scanning
- **Nessus**: Commercial vulnerability scanner
- **OpenVAS**: Open-source vulnerability scanner
- **AWS Inspector**: AWS cloud vulnerability scanning
- **Azure Security Center**: Azure cloud security monitoring

**Nmap Example**:
```bash
# Scan for open ports
nmap -sV -sC -p- 192.168.1.1

# Output
# PORT     STATE SERVICE  VERSION
# 22/tcp   open  ssh     OpenSSH 8.4p1
# 80/tcp   open  http    nginx 1.18.0
```

**AWS Inspector Example**:
```bash
# Install AWS CLI
pip install awscli

# Run vulnerability scan
aws inspector start-assessment-run \
  --assessment-template-arn arn:aws:inspector:region:us-east-1:123456789012:target/0-n0CpHuj8EXAMPLE \
  --assessment-name MyAssessment
```

---

## CI/CD INTEGRATION

### 1. Automated Scanning Pipeline

**GitHub Actions Example**:
```yaml
name: Security Scanning

on: [push, pull_request]

jobs:
  sast:
    runs-on: ubuntu-latest
    steps:
    - uses: actions/checkout@v3

    - name: Install dependencies
      run: composer install

    - name: Run SAST scan
      run: vendor/bin/phpstan analyse src --error-format=json > sast-report.json

    - name: Upload SAST results
      uses: github/codeql-action/upload-sarif@v2
      with:
        sarif_file: 'sast-report.json'

  dependency-scan:
    runs-on: ubuntu-latest
    steps:
    - uses: actions/checkout@v3

    - name: Run dependency scan
      run: composer audit --format=json > audit-report.json

    - name: Upload dependency results
      uses: github/codeql-action/upload-sarif@v2
      with:
        sarif_file: 'audit-report.json'

  container-scan:
    runs-on: ubuntu-latest
    steps:
    - uses: actions/checkout@v3

    - name: Build Docker image
      run: docker build -t myapp:${{ github.sha }} .

    - name: Scan image
      uses: aquasecurity/trivy-action@master
      with:
        image-ref: myapp:${{ github.sha }}
        format: 'sarif'
        severity: 'CRITICAL,HIGH'

    - name: Upload scan results
      uses: github/codeql-action/upload-sarif@v2
      with:
        sarif_file: 'trivy-results.sarif'
```

---

## VULNERABILITY MANAGEMENT

### 1. Vulnerability Lifecycle

```
Discovery → Triage → Remediation → Verification → Closure
```

**Triage Process**:
1. **Assess Severity**:
   - CRITICAL: Immediate action required
   - HIGH: Action required within 7 days
   - MEDIUM: Action required within 30 days
   - LOW: Action required within 90 days

2. **Assess Exploitability**:
   - Public exploit available
   - Proof of concept exists
   - Theoretical only

3. **Assess Impact**:
   - Data breach
   - System compromise
   - Service disruption
   - Information disclosure

---

### 2. Vulnerability Tracking

**Database Schema**:
```sql
CREATE TABLE vulnerabilities (
    id SERIAL PRIMARY KEY,
    cve_id VARCHAR(50),
    severity ENUM('CRITICAL', 'HIGH', 'MEDIUM', 'LOW'),
    status ENUM('OPEN', 'IN_PROGRESS', 'FIXED', 'VERIFIED', 'CLOSED'),
    discovered_at TIMESTAMP,
    assigned_to VARCHAR(255),
    fixed_at TIMESTAMP,
    verified_at TIMESTAMP,
    affected_components TEXT,
    remediation_plan TEXT
);
```

**API Example**:
```python
class VulnerabilityAPI:
    def create_vulnerability(self, vuln_data: dict) -> dict:
        vulnerability = {
            'cve_id': vuln_data['cve_id'],
            'severity': self.assess_severity(vuln_data),
            'status': 'OPEN',
            'discovered_at': datetime.utcnow(),
            'affected_components': vuln_data['components']
        }

        return self.database.insert('vulnerabilities', vulnerability)

    def assign_vulnerability(self, vuln_id: int, assignee: str) -> bool:
        return self.database.update('vulnerabilities', vuln_id, {
            'assigned_to': assignee,
            'status': 'IN_PROGRESS'
        })

    def mark_fixed(self, vuln_id: int, fix_details: dict) -> bool:
        return self.database.update('vulnerabilities', vuln_id, {
            'status': 'FIXED',
            'fixed_at': datetime.utcnow(),
            'remediation_plan': fix_details
        })

    def verify_fix(self, vuln_id: int, verified: bool) -> bool:
        status = 'VERIFIED' if verified else 'OPEN'
        return self.database.update('vulnerabilities', vuln_id, {
            'status': status,
            'verified_at': datetime.utcnow() if verified else None
        })
```

---

## REMEDIATION STRATEGIES

### 1. Patch Management

**Process**:
1. Identify all vulnerable components
2. Check for available patches/updates
3. Test patches in staging
4. Deploy to production
5. Verify fix

**Example**:
```bash
# Update PHP package
composer update vendor/package --with-dependencies

# Update npm package
npm update package

# Update Python package
pip install --upgrade package
```

### 2. Configuration Hardening

**Process**:
1. Review insecure configurations
2. Apply security best practices
3. Test configuration
4. Deploy to production

**Example**:
```bash
# Disable PHP exposion
echo "expose_php = Off" >> /etc/php/8.2/fpm/php.ini

# Update Nginx configuration
sed -i 's/server_tokens on/server_tokens off/' /etc/nginx/nginx.conf
```

### 3. Code Remediation

**Process**:
1. Identify vulnerable code patterns
2. Implement secure alternatives
3. Add unit tests
4. Run security scan verification

---

## VULNERABILITY CHECKLIST

### Before Deployment
- [ ] SAST scan completed and passed
- [ ] DAST scan completed and passed
- [ ] Dependency scan completed and passed
- [ ] Container image scan completed and passed
- [ ] Infrastructure scan completed and passed
- [ ] High/critical vulnerabilities remediated
- [ ] Vulnerability tracking system in place
- [ ] Incident response procedures documented
- [ ] Security team notified of findings

### Ongoing
- [ ] Regular vulnerability scans (weekly/monthly)
- [ ] Automated scanning in CI/CD pipeline
- [ ] Vulnerability backlog managed
- [ ] Remediation timelines enforced
- [ ] Security patches prioritized
- [ ] Vulnerability metrics tracked (MTTR - Mean Time to Remediate)

---

## CROSS-REFERENCES
- For OWASP Top 10: @owasp/OWASP-TOP10.md
- For security patterns: @security/AUTHENTICATION.md, @security/INPUT-VALIDATION.md
- For container security: @devops-engineering/security/CONTAINER-SECURITY.md
