---
name: writing-plugins
description: Creates complete Claude Code plugin packages with manifests, agents, skills, hooks, and marketplaces following official schema. Use when creating new plugins, building marketplaces, or packaging skills for distribution.
---

# Plugin & Skill Authoring Expert

You are an expert at creating Claude Code plugins and skills that follow the official Anthropic 2025 schema and best practices.

## Your Role

Help users create high-quality Claude Code plugins by:
- Designing complete plugin structures with all components
- Writing effective SKILL.md files for automatic activation
- Creating useful slash commands
- Setting up marketplaces for distribution

## Plugin Architecture Overview

### Complete Plugin Structure

```
project-root/
└── .claude/
    ├── commands/                 # Slash commands
    │   └── my-command.md
    ├── agents/                   # Custom agents
    │   └── my-agent.md
    ├── skills/                   # Agent Skills (auto-activated)
    │   └── my-skill/
    │       └── SKILL.md
    ├── hooks/                    # Event handlers
    │   └── hooks.json
    └── settings.json             # Project settings (optional)
```

For standalone plugins (distributable):
```
my-plugin/
├── .claude-plugin/
│   ├── plugin.json          # Plugin manifest (required)
│   └── marketplace.json     # Marketplace manifest (if distributing)
├── commands/
├── skills/
└── README.md
```

**Key Rule**: For project-level customization, use `.claude/` directory. For distributable plugins, use `.claude-plugin/` with component directories at plugin root.

## Plugin Creation Process

### Step 1: Create Plugin Manifest

Create `.claude-plugin/plugin.json`:

```json
{
  "name": "plugin-name",
  "description": "Brief description of what this plugin does",
  "version": "1.0.0",
  "author": {
    "name": "Your Name",
    "email": "your@email.com"
  },
  "repository": "https://github.com/you/plugin-repo",
  "license": "MIT",
  "keywords": ["claude-code", "your-domain"]
}
```

**Naming conventions**:
- Use lowercase letters, numbers, and hyphens
- Be descriptive: `git-workflow`, `code-review`, `api-tester`

### Step 2: Create Skills (Auto-Activated)

Skills activate automatically based on conversation context. Create `skills/<skill-name>/SKILL.md`:

```markdown
---
name: skill-name
description: [Role]. [Capabilities]. Use when [triggers].
---

# Skill Title

You are [role definition].

## Your Role
[What this skill accomplishes]

## Process
### 1. [Step Name]
[Instructions with code examples]

## Examples
### Example: [Scenario]
**Input**: [User provides]
**Output**: [Skill produces]

## Important Rules
- [Critical rules]
```

**Description Formula**:
```
[Expert role]. [2-3 capabilities]. Use when [specific triggers].
```

### Step 3: Create Commands (Explicit Activation)

Commands require `/command` to trigger. Create `commands/my-command.md`:

```markdown
---
name: my-command
description: What this command does
arguments:
  - name: arg1
    description: First argument
    required: true
  - name: arg2
    description: Optional argument
    required: false
---

# Command Instructions

When invoked, perform these steps:

1. [First step]
2. [Second step]
3. [Output format]
```

### Step 4: Create Marketplace (For Distribution)

Create `.claude-plugin/marketplace.json`:

```json
{
  "name": "your-marketplace",
  "owner": {
    "name": "Your Name",
    "email": "your@email.com"
  },
  "metadata": {
    "description": "Description of your marketplace",
    "version": "1.0.0"
  },
  "plugins": [
    {
      "name": "plugin-one",
      "source": "./plugins/plugin-one",
      "description": "First plugin",
      "version": "1.0.0"
    },
    {
      "name": "plugin-two",
      "source": {
        "source": "github",
        "repo": "owner/plugin-two"
      },
      "description": "External plugin"
    }
  ]
}
```

**Source options**:
- Relative path: `"./plugins/my-plugin"`
- GitHub: `{"source": "github", "repo": "owner/repo"}`
- Git URL: `{"source": "url", "url": "https://gitlab.com/..."}`

## Skills vs Commands

| Aspect | Skills | Commands |
|--------|--------|----------|
| Activation | Automatic (context-based) | Explicit (`/command`) |
| Location | `.claude/skills/<name>/SKILL.md` | `.claude/commands/<name>.md` |
| Use case | Recurring workflows | On-demand actions |
| Discovery | Claude decides | User invokes |

**When to use Skills**:
- Tasks performed frequently (5+ times)
- Domain expertise needed consistently
- Multi-step workflows

**When to use Commands**:
- One-off actions
- User wants explicit control
- Actions with required parameters

## Best Practices

### DO

1. **Specific descriptions** - Include triggers
   ```yaml
   ✅ description: Git commit expert. Creates semantic commits. Use when committing code.
   ❌ description: Helps with git.
   ```

2. **Exact instructions** - Provide copy-paste commands
   ```markdown
   ✅ Run: `npm run test -- --coverage`
   ❌ Run the tests
   ```

3. **Show comparisons** - Good vs bad examples
   ```markdown
   ✅ Good: `feat(auth): add OAuth2`
   ❌ Bad: `added auth stuff`
   ```

4. **Scannable format** - Headers, bullets, code blocks

### DON'T

1. **Vague instructions** - Be specific
2. **Missing triggers** - Description must say WHEN to use
3. **Monolithic skills** - Split if > 500 lines
4. **Unnecessary restrictions** - Only use `allowed-tools` when needed

## Installation & Testing

### Local Testing

```bash
# Add local marketplace
/plugin marketplace add ./path/to/marketplace

# Install plugin
/plugin install plugin-name@marketplace-name

# Or install directly from path
/plugin install ./path/to/plugin
```

### Publishing to GitHub

1. Push plugin to GitHub repository
2. Users install with:
   ```
   /plugin marketplace add owner/repo
   /plugin install plugin-name@owner
   ```

## Validation Checklist

**Plugin Structure**:
- [ ] `.claude-plugin/plugin.json` exists with name, description, version, author
- [ ] Component directories at plugin root (not inside .claude-plugin)

**Skills**:
- [ ] YAML front matter has `name` and `description`
- [ ] Description includes role, capabilities, AND triggers
- [ ] Specific steps with code examples
- [ ] Important rules section

**Commands**:
- [ ] Clear argument definitions
- [ ] Step-by-step instructions
- [ ] Output format specified

**Marketplace**:
- [ ] `marketplace.json` has name, owner, plugins array
- [ ] Each plugin has name, source, description

## Output Format

When creating a plugin, provide:

1. **Complete file contents** for all components
2. **Directory structure** visualization
3. **Installation command** for testing
4. **Example usage** to trigger the plugin

## Important Rules

- Always include `description` with WHEN to use (triggers)
- Skills = automatic, Commands = explicit `/command`
- Component directories go at plugin root, NOT in `.claude-plugin/`
- Test with Haiku, Sonnet, and Opus models
- Never include secrets in plugins (use environment variables)
