/*
 * CSV Data Output Pattern Example
 * 
 * Demonstrates structured data logging in CSV format with:
 * - Timestamp (milliseconds)
 * - Multiple sensor channels (temperature, humidity, pressure)
 * - F() macro for PROGMEM string storage
 * - Proper CSV formatting with headers
 * 
 * Generated by: arduino-code-generator
 * Pattern: CSV Data Formatting
 * License: MIT
 */

// === Configuration ===
const uint16_t SAMPLE_INTERVAL_MS = 1000; // 1 second
const uint8_t TEMP_PIN = A0;
const uint8_t HUMIDITY_PIN = A1;
const uint8_t PRESSURE_PIN = A2;

// === Simulated Sensor Class ===
class SensorSimulator {
private:
  float baseValue;
  float noiseLevel;
  
public:
  SensorSimulator(float base, float noise) 
    : baseValue(base), noiseLevel(noise) {}
  
  float read() {
    // Simulate sensor with noise and slow drift
    float drift = sin(millis() / 10000.0) * (noiseLevel * 2);
    float noise = (random(-100, 100) / 100.0) * noiseLevel;
    return baseValue + drift + noise;
  }
};

// === Sensor Instances ===
SensorSimulator tempSensor(22.5, 0.5);      // 22.5°C ± 0.5°C
SensorSimulator humiditySensor(55.0, 2.0);  // 55% ± 2%
SensorSimulator pressureSensor(1013.25, 1.0); // 1013.25 hPa ± 1 hPa

// === CSV Output Functions ===
void printCSVHeader() {
  Serial.println(F("timestamp_ms,temperature_c,humidity_percent,pressure_hpa"));
}

void printCSVRow(unsigned long timestamp, float temp, float humidity, float pressure) {
  Serial.print(timestamp);
  Serial.print(F(","));
  Serial.print(temp, 2);
  Serial.print(F(","));
  Serial.print(humidity, 1);
  Serial.print(F(","));
  Serial.println(pressure, 2);
}

void setup() {
  Serial.begin(115200);
  while (!Serial && millis() < 3000);
  
  randomSeed(analogRead(A3)); // Seed for simulation
  
  Serial.println(F("\n=== CSV Data Logger Example ==="));
  Serial.println(F("Simulated environmental monitoring"));
  Serial.println(F("Copy output to .csv file for analysis\n"));
  
  delay(500);
  printCSVHeader();
}

void loop() {
  static unsigned long lastSample = 0;
  static uint16_t sampleCount = 0;
  
  unsigned long now = millis();
  
  if (now - lastSample >= SAMPLE_INTERVAL_MS) {
    lastSample = now;
    
    // Read sensors
    float temperature = tempSensor.read();
    float humidity = humiditySensor.read();
    float pressure = pressureSensor.read();
    
    // Output CSV row
    printCSVRow(now, temperature, humidity, pressure);
    
    sampleCount++;
    
    // Optional: Stop after 60 samples (1 minute)
    if (sampleCount >= 60) {
      Serial.println(F("\n# Data logging complete (60 samples)"));
      while (1); // Halt
    }
  }
}
