/*
 * Hardware Detection Pattern Example
 * 
 * Demonstrates runtime board detection and capability reporting:
 * - Compile-time board identification
 * - Memory capacity detection
 * - Clock speed determination
 * - Feature flags (WiFi, Bluetooth, etc.)
 * 
 * Generated by: arduino-code-generator
 * Pattern: Hardware Detection
 * License: MIT
 */

// === Board Detection ===
struct BoardInfo {
  const char* name;
  const char* mcu;
  uint32_t flashSize;
  uint32_t sramSize;
  uint32_t clockSpeed;
  uint8_t adcBits;
  bool hasWiFi;
  bool hasBluetooth;
  uint32_t serialBaud;
};

BoardInfo detectBoard() {
  BoardInfo info;
  
  #if defined(ARDUINO_AVR_UNO)
    info.name = "Arduino UNO";
    info.mcu = "ATmega328P";
    info.flashSize = 32768;
    info.sramSize = 2048;
    info.clockSpeed = 16000000;
    info.adcBits = 10;
    info.hasWiFi = false;
    info.hasBluetooth = false;
    info.serialBaud = 9600;
    
  #elif defined(ARDUINO_AVR_MEGA2560)
    info.name = "Arduino Mega 2560";
    info.mcu = "ATmega2560";
    info.flashSize = 262144;
    info.sramSize = 8192;
    info.clockSpeed = 16000000;
    info.adcBits = 10;
    info.hasWiFi = false;
    info.hasBluetooth = false;
    info.serialBaud = 9600;
    
  #elif defined(ESP32)
    info.name = "ESP32";
    info.mcu = "ESP32 Dual-Core";
    info.flashSize = 4194304; // Typical 4MB
    info.sramSize = 520000;
    info.clockSpeed = 240000000;
    info.adcBits = 12;
    info.hasWiFi = true;
    info.hasBluetooth = true;
    info.serialBaud = 115200;
    
  #elif defined(ARDUINO_ARCH_RP2040)
    info.name = "Raspberry Pi Pico (RP2040)";
    info.mcu = "RP2040 Dual-Core ARM";
    info.flashSize = 2097152; // 2MB
    info.sramSize = 264000;
    info.clockSpeed = 133000000;
    info.adcBits = 12;
    info.hasWiFi = false;
    info.hasBluetooth = false;
    info.serialBaud = 115200;
    
  #elif defined(ARDUINO_SAMD_ZERO)
    info.name = "Arduino Zero";
    info.mcu = "SAMD21";
    info.flashSize = 262144;
    info.sramSize = 32768;
    info.clockSpeed = 48000000;
    info.adcBits = 12;
    info.hasWiFi = false;
    info.hasBluetooth = false;
    info.serialBaud = 115200;
    
  #else
    info.name = "Unknown Board";
    info.mcu = "Unknown";
    info.flashSize = 0;
    info.sramSize = 0;
    info.clockSpeed = 0;
    info.adcBits = 10;
    info.hasWiFi = false;
    info.hasBluetooth = false;
    info.serialBaud = 9600;
  #endif
  
  return info;
}

// === Helper Functions ===
void printBoardInfo(const BoardInfo& info) {
  Serial.println(F("\n=== Hardware Detection Report ===\n"));
  
  Serial.print(F("Board: "));
  Serial.println(info.name);
  
  Serial.print(F("MCU: "));
  Serial.println(info.mcu);
  
  Serial.print(F("Flash Memory: "));
  Serial.print(info.flashSize / 1024);
  Serial.println(F(" KB"));
  
  Serial.print(F("SRAM: "));
  Serial.print(info.sramSize);
  Serial.println(F(" bytes"));
  
  Serial.print(F("Clock Speed: "));
  Serial.print(info.clockSpeed / 1000000);
  Serial.println(F(" MHz"));
  
  Serial.print(F("ADC Resolution: "));
  Serial.print(info.adcBits);
  Serial.println(F(" bits"));
  
  Serial.println(F("\nCapabilities:"));
  Serial.print(F("  WiFi: "));
  Serial.println(info.hasWiFi ? F("Yes") : F("No"));
  Serial.print(F("  Bluetooth: "));
  Serial.println(info.hasBluetooth ? F("Yes") : F("No"));
  
  Serial.println(F("\nCompile-Time Defines:"));
  #ifdef ARDUINO_AVR_UNO
    Serial.println(F("  - ARDUINO_AVR_UNO"));
  #endif
  #ifdef ESP32
    Serial.println(F("  - ESP32"));
  #endif
  #ifdef ARDUINO_ARCH_RP2040
    Serial.println(F("  - ARDUINO_ARCH_RP2040"));
  #endif
  
  Serial.print(F("\nArduino IDE Version: "));
  Serial.println(ARDUINO);
}

void printMemoryUsage() {
  Serial.println(F("\n=== Memory Usage ==="));
  
  #if defined(ARDUINO_AVR_UNO) || defined(ARDUINO_AVR_MEGA2560)
    // AVR can report free RAM
    extern int __heap_start, *__brkval;
    int freeRam = (int)&freeRam - (__brkval == 0 ? (int)&__heap_start : (int)__brkval);
    Serial.print(F("Free SRAM: "));
    Serial.print(freeRam);
    Serial.println(F(" bytes"));
  #else
    Serial.println(F("(Memory reporting not available on this platform)"));
  #endif
}

void setup() {
  BoardInfo board = detectBoard();
  
  Serial.begin(board.serialBaud);
  while (!Serial && millis() < 3000);
  
  printBoardInfo(board);
  printMemoryUsage();
  
  Serial.println(F("\n=== Board detection complete ==="));
}

void loop() {
  // Periodically print memory usage
  static unsigned long lastReport = 0;
  if (millis() - lastReport >= 10000) {
    lastReport = millis();
    printMemoryUsage();
  }
}
