/*
 * Non-Blocking Task Scheduler Pattern Example
 * 
 * Demonstrates cooperative multitasking without delay() using:
 * - Lightweight task scheduler (millis() based)
 * - Multiple independent tasks with different intervals
 * - Task priority and execution tracking
 * - No blocking delays - maintains responsiveness
 * 
 * Generated by: arduino-code-generator
 * Pattern: Non-Blocking Task Scheduler
 * License: MIT
 */

// === Task Scheduler ===
class Task {
public:
  using TaskFunction = void (*)();
  
private:
  TaskFunction function;
  unsigned long interval;
  unsigned long lastRun;
  bool enabled;
  const char* name;
  uint32_t executionCount;
  
public:
  Task(TaskFunction func, unsigned long intervalMs, const char* taskName)
    : function(func), interval(intervalMs), lastRun(0), 
      enabled(true), name(taskName), executionCount(0) {}
  
  void update() {
    if (!enabled) return;
    
    unsigned long now = millis();
    if (now - lastRun >= interval) {
      lastRun = now;
      function();
      executionCount++;
    }
  }
  
  void setEnabled(bool state) { enabled = state; }
  void setInterval(unsigned long ms) { interval = ms; }
  
  bool isEnabled() const { return enabled; }
  const char* getName() const { return name; }
  uint32_t getExecutionCount() const { return executionCount; }
  unsigned long getInterval() const { return interval; }
};

// === Task Scheduler Manager ===
template<size_t MAX_TASKS>
class Scheduler {
private:
  Task* tasks[MAX_TASKS];
  size_t taskCount = 0;
  
public:
  bool addTask(Task* task) {
    if (taskCount >= MAX_TASKS) return false;
    tasks[taskCount++] = task;
    return true;
  }
  
  void run() {
    for (size_t i = 0; i < taskCount; i++) {
      tasks[i]->update();
    }
  }
  
  void printStatus() {
    Serial.println(F("\n=== Task Status ==="));
    for (size_t i = 0; i < taskCount; i++) {
      Serial.print(F("Task: "));
      Serial.print(tasks[i]->getName());
      Serial.print(F(" | Interval: "));
      Serial.print(tasks[i]->getInterval());
      Serial.print(F("ms | Executions: "));
      Serial.print(tasks[i]->getExecutionCount());
      Serial.print(F(" | Status: "));
      Serial.println(tasks[i]->isEnabled() ? F("ENABLED") : F("DISABLED"));
    }
    Serial.println();
  }
};

// === Task Functions ===
void taskFast() {
  static uint16_t counter = 0;
  Serial.print(F("[FAST] Execution #"));
  Serial.println(++counter);
}

void taskMedium() {
  Serial.print(F("[MEDIUM] Sensor reading: "));
  Serial.println(analogRead(A0));
}

void taskSlow() {
  Serial.print(F("[SLOW] Uptime: "));
  Serial.print(millis() / 1000);
  Serial.println(F(" seconds"));
}

void taskHeartbeat() {
  static bool ledState = false;
  ledState = !ledState;
  digitalWrite(LED_BUILTIN, ledState);
  Serial.println(ledState ? F("[LED] ON") : F("[LED] OFF"));
}

void taskReport() {
  scheduler.printStatus();
}

// === Task Definitions ===
Task fastTask(taskFast, 500, "Fast Task");
Task mediumTask(taskMedium, 2000, "Medium Task");
Task slowTask(taskSlow, 5000, "Slow Task");
Task heartbeatTask(taskHeartbeat, 1000, "Heartbeat LED");
Task reportTask(taskReport, 15000, "Status Report");

// === Scheduler Instance ===
Scheduler<5> scheduler;

void setup() {
  Serial.begin(115200);
  pinMode(LED_BUILTIN, OUTPUT);
  
  Serial.println(F("\n=== Non-Blocking Scheduler Example ==="));
  Serial.println(F("Running 5 independent tasks:"));
  Serial.println(F("- Fast: 500ms (counter)"));
  Serial.println(F("- Medium: 2000ms (sensor read)"));
  Serial.println(F("- Slow: 5000ms (uptime)"));
  Serial.println(F("- Heartbeat: 1000ms (LED toggle)"));
  Serial.println(F("- Report: 15000ms (status)\n"));
  
  // Register tasks
  scheduler.addTask(&fastTask);
  scheduler.addTask(&mediumTask);
  scheduler.addTask(&slowTask);
  scheduler.addTask(&heartbeatTask);
  scheduler.addTask(&reportTask);
  
  Serial.println(F("Scheduler started...\n"));
}

void loop() {
  scheduler.run();
  
  // Main loop remains responsive for other operations
  // (e.g., button checks, serial input, etc.)
}
