/*
 * Debounced Button Pattern Example
 * 
 * Demonstrates robust button handling with:
 * - Debouncing (noise rejection)
 * - Press/release event detection
 * - Long-press detection (1 second threshold)
 * - Non-blocking state machine implementation
 * 
 * Generated by: arduino-code-generator
 * Pattern: Button Input with Debouncing
 * License: MIT
 */

// === Configuration ===
#if defined(ESP32)
  const uint8_t BUTTON_PIN = 4;
  const uint32_t SERIAL_BAUD = 115200;
#elif defined(ARDUINO_ARCH_RP2040)
  const uint8_t BUTTON_PIN = 10;
  const uint32_t SERIAL_BAUD = 115200;
#else
  const uint8_t BUTTON_PIN = 2;
  const uint32_t SERIAL_BAUD = 9600;
#endif

// === Debounced Button Class ===
class DebouncedButton {
public:
  enum Event {
    NONE,
    PRESSED,
    RELEASED,
    LONG_PRESS
  };
  
private:
  const uint8_t pin;
  const uint16_t DEBOUNCE_MS = 50;
  const uint16_t LONG_PRESS_MS = 1000;
  
  bool lastReading = HIGH;
  bool lastStableState = HIGH;
  unsigned long lastDebounceTime = 0;
  unsigned long pressStartTime = 0;
  bool longPressTriggered = false;
  
public:
  DebouncedButton(uint8_t buttonPin) : pin(buttonPin) {}
  
  void begin() {
    pinMode(pin, INPUT_PULLUP);
  }
  
  Event update() {
    bool reading = digitalRead(pin);
    unsigned long now = millis();
    
    // Reset debounce timer on state change
    if (reading != lastReading) {
      lastDebounceTime = now;
    }
    lastReading = reading;
    
    // Wait for stable state
    if ((now - lastDebounceTime) < DEBOUNCE_MS) {
      return NONE;
    }
    
    // State changed
    if (reading != lastStableState) {
      lastStableState = reading;
      
      if (reading == LOW) {
        // Button pressed
        pressStartTime = now;
        longPressTriggered = false;
        return PRESSED;
      } else {
        // Button released
        return longPressTriggered ? NONE : RELEASED;
      }
    }
    
    // Check for long press (while held)
    if (lastStableState == LOW && !longPressTriggered) {
      if ((now - pressStartTime) >= LONG_PRESS_MS) {
        longPressTriggered = true;
        return LONG_PRESS;
      }
    }
    
    return NONE;
  }
  
  bool isPressed() const { return lastStableState == LOW; }
};

// === Application ===
DebouncedButton button(BUTTON_PIN);
uint16_t pressCount = 0;
uint16_t longPressCount = 0;

void setup() {
  Serial.begin(SERIAL_BAUD);
  button.begin();
  
  Serial.println(F("\n=== Debounced Button Example ==="));
  Serial.println(F("Events detected:"));
  Serial.println(F("- PRESS: Short button press"));
  Serial.println(F("- RELEASE: Button released"));
  Serial.println(F("- LONG_PRESS: Hold for 1 second"));
  Serial.println();
}

void loop() {
  DebouncedButton::Event event = button.update();
  
  switch (event) {
    case DebouncedButton::PRESSED:
      Serial.println(F("-> PRESSED"));
      pressCount++;
      break;
      
    case DebouncedButton::RELEASED:
      Serial.print(F("-> RELEASED (count: "));
      Serial.print(pressCount);
      Serial.println(F(")"));
      break;
      
    case DebouncedButton::LONG_PRESS:
      Serial.println(F("-> LONG_PRESS"));
      longPressCount++;
      Serial.print(F("   Long press count: "));
      Serial.println(longPressCount);
      break;
      
    case DebouncedButton::NONE:
      // No event
      break;
  }
}
