/*
 * Data Logging Pattern Example
 * 
 * Demonstrates persistent data storage using EEPROM:
 * - Sensor data logging with timestamps
 * - Circular buffer implementation
 * - EEPROM wear leveling (write minimization)
 * - Data retrieval and playback
 * 
 * Generated by: arduino-code-generator
 * Pattern: Data Logging & Persistence
 * License: MIT
 */

#include <EEPROM.h>

// === Configuration ===
const uint16_t LOG_INTERVAL_MS = 5000; // 5 seconds
const uint8_t MAX_LOG_ENTRIES = 20;    // Circular buffer size

// === Log Entry Structure ===
struct LogEntry {
  uint32_t timestamp;  // Milliseconds since boot
  int16_t temperature; // °C * 10 (e.g., 235 = 23.5°C)
  uint16_t humidity;   // % * 10 (e.g., 550 = 55.0%)
  uint16_t lightLevel; // Raw ADC value
};

// === EEPROM Data Logger ===
class EEPROMLogger {
private:
  const uint16_t MAGIC_NUMBER = 0xABCD;
  const uint16_t ADDR_MAGIC = 0;
  const uint16_t ADDR_HEAD = 2;
  const uint16_t ADDR_COUNT = 4;
  const uint16_t ADDR_DATA_START = 6;
  
  uint8_t headIndex;
  uint8_t entryCount;
  
  void writeUint16(uint16_t addr, uint16_t value) {
    EEPROM.write(addr, value & 0xFF);
    EEPROM.write(addr + 1, (value >> 8) & 0xFF);
  }
  
  uint16_t readUint16(uint16_t addr) {
    uint8_t low = EEPROM.read(addr);
    uint8_t high = EEPROM.read(addr + 1);
    return (high << 8) | low;
  }
  
  void writeLogEntry(uint8_t index, const LogEntry& entry) {
    uint16_t addr = ADDR_DATA_START + (index * sizeof(LogEntry));
    
    const uint8_t* data = (const uint8_t*)&entry;
    for (size_t i = 0; i < sizeof(LogEntry); i++) {
      EEPROM.write(addr + i, data[i]);
    }
  }
  
  LogEntry readLogEntry(uint8_t index) {
    uint16_t addr = ADDR_DATA_START + (index * sizeof(LogEntry));
    
    LogEntry entry;
    uint8_t* data = (uint8_t*)&entry;
    for (size_t i = 0; i < sizeof(LogEntry); i++) {
      data[i] = EEPROM.read(addr + i);
    }
    return entry;
  }
  
public:
  void begin() {
    // Check if EEPROM is initialized
    uint16_t magic = readUint16(ADDR_MAGIC);
    
    if (magic != MAGIC_NUMBER) {
      // First-time initialization
      Serial.println(F("Initializing EEPROM logger..."));
      writeUint16(ADDR_MAGIC, MAGIC_NUMBER);
      writeUint16(ADDR_HEAD, 0);
      writeUint16(ADDR_COUNT, 0);
      headIndex = 0;
      entryCount = 0;
    } else {
      // Load existing state
      headIndex = readUint16(ADDR_HEAD);
      entryCount = readUint16(ADDR_COUNT);
      Serial.print(F("Loaded "));
      Serial.print(entryCount);
      Serial.println(F(" existing log entries"));
    }
  }
  
  void logData(const LogEntry& entry) {
    writeLogEntry(headIndex, entry);
    
    headIndex = (headIndex + 1) % MAX_LOG_ENTRIES;
    if (entryCount < MAX_LOG_ENTRIES) entryCount++;
    
    writeUint16(ADDR_HEAD, headIndex);
    writeUint16(ADDR_COUNT, entryCount);
  }
  
  void printAllLogs() {
    Serial.println(F("\n=== Stored Logs ==="));
    Serial.println(F("Index  Timestamp  Temp(°C)  Humidity(%)  Light"));
    Serial.println(F("-----  ---------  --------  -----------  -----"));
    
    for (uint8_t i = 0; i < entryCount; i++) {
      // Read in circular order (oldest to newest)
      uint8_t index = (headIndex - entryCount + i + MAX_LOG_ENTRIES) % MAX_LOG_ENTRIES;
      LogEntry entry = readLogEntry(index);
      
      Serial.print(i);
      Serial.print(F("      "));
      Serial.print(entry.timestamp / 1000);
      Serial.print(F("s      "));
      Serial.print(entry.temperature / 10.0, 1);
      Serial.print(F("      "));
      Serial.print(entry.humidity / 10.0, 1);
      Serial.print(F("      "));
      Serial.println(entry.lightLevel);
    }
    Serial.println();
  }
  
  void clearAll() {
    headIndex = 0;
    entryCount = 0;
    writeUint16(ADDR_HEAD, 0);
    writeUint16(ADDR_COUNT, 0);
    Serial.println(F("All logs cleared"));
  }
  
  uint8_t getCount() const { return entryCount; }
};

// === Application ===
EEPROMLogger logger;

LogEntry readSensors() {
  LogEntry entry;
  entry.timestamp = millis();
  
  // Simulate temperature sensor (22-25°C)
  entry.temperature = random(220, 250);
  
  // Simulate humidity sensor (40-60%)
  entry.humidity = random(400, 600);
  
  // Read real light sensor
  entry.lightLevel = analogRead(A0);
  
  return entry;
}

void setup() {
  Serial.begin(115200);
  while (!Serial && millis() < 3000);
  
  Serial.println(F("\n=== EEPROM Data Logger Example ==="));
  Serial.print(F("EEPROM Size: "));
  Serial.print(EEPROM.length());
  Serial.println(F(" bytes"));
  Serial.print(F("Max Log Entries: "));
  Serial.println(MAX_LOG_ENTRIES);
  Serial.print(F("Entry Size: "));
  Serial.print(sizeof(LogEntry));
  Serial.println(F(" bytes"));
  Serial.println();
  
  logger.begin();
  
  Serial.println(F("Commands:"));
  Serial.println(F("  'p' - Print all logs"));
  Serial.println(F("  'c' - Clear all logs"));
  Serial.println();
}

void loop() {
  static unsigned long lastLog = 0;
  
  // Log data periodically
  if (millis() - lastLog >= LOG_INTERVAL_MS) {
    lastLog = millis();
    
    LogEntry entry = readSensors();
    logger.logData(entry);
    
    Serial.print(F("Logged: "));
    Serial.print(entry.temperature / 10.0, 1);
    Serial.print(F("°C, "));
    Serial.print(entry.humidity / 10.0, 1);
    Serial.print(F("%, Light="));
    Serial.print(entry.lightLevel);
    Serial.print(F(" ("));
    Serial.print(logger.getCount());
    Serial.println(F(" total)"));
  }
  
  // Handle serial commands
  if (Serial.available()) {
    char cmd = Serial.read();
    
    if (cmd == 'p') {
      logger.printAllLogs();
    } else if (cmd == 'c') {
      logger.clearAll();
    }
  }
}
