/*
 * ADC Filtering Pattern Example
 * 
 * Demonstrates multiple filtering techniques for noisy sensor readings:
 * - Moving average filter (smoothing)
 * - Exponential moving average (EMA)
 * - Median filter (spike rejection)
 * 
 * Generated by: arduino-code-generator
 * Pattern: Filtering & Signal Processing
 * License: MIT
 */

// === Configuration ===
const uint8_t SENSOR_PIN = A0;
const uint16_t SAMPLE_INTERVAL_MS = 100;

// === Moving Average Filter ===
template<size_t N>
class MovingAverage {
  float buffer[N] = {0};
  size_t index = 0;
  size_t count = 0;
  
public:
  float update(float value) {
    buffer[index] = value;
    index = (index + 1) % N;
    if (count < N) count++;
    
    float sum = 0;
    for (size_t i = 0; i < count; i++) {
      sum += buffer[i];
    }
    return sum / count;
  }
  
  void reset() {
    count = 0;
    index = 0;
  }
};

// === Exponential Moving Average ===
class EMA {
  float alpha;
  float value;
  bool initialized = false;
  
public:
  EMA(float smoothing = 0.1) : alpha(smoothing) {}
  
  float update(float newValue) {
    if (!initialized) {
      value = newValue;
      initialized = true;
      return value;
    }
    
    value = alpha * newValue + (1.0 - alpha) * value;
    return value;
  }
  
  void reset() {
    initialized = false;
  }
};

// === Median Filter (3-sample) ===
class MedianFilter {
  float buffer[3] = {0};
  size_t index = 0;
  
public:
  float update(float value) {
    buffer[index] = value;
    index = (index + 1) % 3;
    
    // Simple 3-value sort
    float sorted[3];
    memcpy(sorted, buffer, sizeof(buffer));
    
    if (sorted[0] > sorted[1]) { float t = sorted[0]; sorted[0] = sorted[1]; sorted[1] = t; }
    if (sorted[1] > sorted[2]) { float t = sorted[1]; sorted[1] = sorted[2]; sorted[2] = t; }
    if (sorted[0] > sorted[1]) { float t = sorted[0]; sorted[0] = sorted[1]; sorted[1] = t; }
    
    return sorted[1]; // Return median
  }
};

// === Filter Instances ===
MovingAverage<10> movingAvg;
EMA ema(0.15);
MedianFilter medianFilter;

void setup() {
  Serial.begin(115200);
  pinMode(SENSOR_PIN, INPUT);
  
  Serial.println(F("\n=== ADC Filtering Example ==="));
  Serial.println(F("Comparing 4 filtering techniques:"));
  Serial.println(F("RAW, MovingAvg(10), EMA(0.15), Median(3)"));
  Serial.println();
}

void loop() {
  static unsigned long lastSample = 0;
  
  if (millis() - lastSample >= SAMPLE_INTERVAL_MS) {
    lastSample = millis();
    
    // Read raw sensor value
    int rawADC = analogRead(SENSOR_PIN);
    float rawVoltage = rawADC * (5.0 / 1023.0);
    
    // Apply filters
    float avgFiltered = movingAvg.update(rawVoltage);
    float emaFiltered = ema.update(rawVoltage);
    float medianFiltered = medianFilter.update(rawVoltage);
    
    // Output CSV format
    Serial.print(rawVoltage, 3);
    Serial.print(F(","));
    Serial.print(avgFiltered, 3);
    Serial.print(F(","));
    Serial.print(emaFiltered, 3);
    Serial.print(F(","));
    Serial.println(medianFiltered, 3);
  }
}
