/*
 * Finite State Machine Pattern Example
 * 
 * Demonstrates FSM implementation for traffic light control:
 * - Explicit state enumeration
 * - State transition logic
 * - Timing-based state changes (non-blocking)
 * - State-specific behavior and outputs
 * 
 * Generated by: arduino-code-generator
 * Pattern: State Machine
 * License: MIT
 */

// === Configuration ===
#if defined(ESP32)
  const uint8_t RED_LED = 25;
  const uint8_t YELLOW_LED = 26;
  const uint8_t GREEN_LED = 27;
#else
  const uint8_t RED_LED = 11;
  const uint8_t YELLOW_LED = 10;
  const uint8_t GREEN_LED = 9;
#endif

// === State Machine Definition ===
enum TrafficLightState {
  STATE_RED,
  STATE_RED_YELLOW,
  STATE_GREEN,
  STATE_YELLOW
};

class TrafficLight {
private:
  TrafficLightState currentState;
  unsigned long stateStartTime;
  
  // State durations (milliseconds)
  static const unsigned long RED_DURATION = 5000;
  static const unsigned long RED_YELLOW_DURATION = 2000;
  static const unsigned long GREEN_DURATION = 5000;
  static const unsigned long YELLOW_DURATION = 2000;
  
  void setLEDs(bool red, bool yellow, bool green) {
    digitalWrite(RED_LED, red ? HIGH : LOW);
    digitalWrite(YELLOW_LED, yellow ? HIGH : LOW);
    digitalWrite(GREEN_LED, green ? HIGH : LOW);
  }
  
  const char* getStateName(TrafficLightState state) {
    switch (state) {
      case STATE_RED: return "RED";
      case STATE_RED_YELLOW: return "RED+YELLOW";
      case STATE_GREEN: return "GREEN";
      case STATE_YELLOW: return "YELLOW";
      default: return "UNKNOWN";
    }
  }
  
  void enterState(TrafficLightState newState) {
    currentState = newState;
    stateStartTime = millis();
    
    Serial.print(F("State: "));
    Serial.print(getStateName(newState));
    Serial.print(F(" (duration: "));
    
    // Set LEDs and print duration
    switch (newState) {
      case STATE_RED:
        setLEDs(true, false, false);
        Serial.print(RED_DURATION / 1000);
        break;
        
      case STATE_RED_YELLOW:
        setLEDs(true, true, false);
        Serial.print(RED_YELLOW_DURATION / 1000);
        break;
        
      case STATE_GREEN:
        setLEDs(false, false, true);
        Serial.print(GREEN_DURATION / 1000);
        break;
        
      case STATE_YELLOW:
        setLEDs(false, true, false);
        Serial.print(YELLOW_DURATION / 1000);
        break;
    }
    
    Serial.println(F("s)"));
  }
  
public:
  TrafficLight() : currentState(STATE_RED), stateStartTime(0) {}
  
  void begin() {
    pinMode(RED_LED, OUTPUT);
    pinMode(YELLOW_LED, OUTPUT);
    pinMode(GREEN_LED, OUTPUT);
    
    enterState(STATE_RED);
  }
  
  void update() {
    unsigned long elapsed = millis() - stateStartTime;
    
    // State transition logic
    switch (currentState) {
      case STATE_RED:
        if (elapsed >= RED_DURATION) {
          enterState(STATE_RED_YELLOW);
        }
        break;
        
      case STATE_RED_YELLOW:
        if (elapsed >= RED_YELLOW_DURATION) {
          enterState(STATE_GREEN);
        }
        break;
        
      case STATE_GREEN:
        if (elapsed >= GREEN_DURATION) {
          enterState(STATE_YELLOW);
        }
        break;
        
      case STATE_YELLOW:
        if (elapsed >= YELLOW_DURATION) {
          enterState(STATE_RED);
        }
        break;
    }
  }
  
  TrafficLightState getState() const {
    return currentState;
  }
};

// === Application ===
TrafficLight trafficLight;

void setup() {
  Serial.begin(115200);
  
  Serial.println(F("\n=== Traffic Light FSM Example ==="));
  Serial.println(F("Cycle: RED -> RED+YELLOW -> GREEN -> YELLOW"));
  Serial.println(F("Pin Configuration:"));
  Serial.print(F("  RED LED: Pin "));
  Serial.println(RED_LED);
  Serial.print(F("  YELLOW LED: Pin "));
  Serial.println(YELLOW_LED);
  Serial.print(F("  GREEN LED: Pin "));
  Serial.println(GREEN_LED);
  Serial.println();
  
  trafficLight.begin();
}

void loop() {
  trafficLight.update();
  
  // Main loop remains responsive for other operations
}
