#!/bin/bash

# Read JSON input from stdin
input=$(cat)

# Extract current directory from JSON
cwd=$(echo "$input" | jq -r '.workspace.current_dir')

# Assign workspace emoji based on path
get_workspace_emoji() {
    local path="$1"
    local basename=$(basename "$path")

    # Semantic mapping for common project types
    case "$basename" in
        *caro*) echo "🦀" ;; # Rust crab
        *rust*) echo "🦀" ;;
        *node*|*npm*|*react*|*next*) echo "📦" ;;
        *python*|*py*) echo "🐍" ;;
        *go*|*golang*) echo "🐹" ;;
        *java*) echo "☕" ;;
        *docker*) echo "🐳" ;;
        *web*|*site*) echo "🌐" ;;
        *api*) echo "🔌" ;;
        *db*|*database*) echo "🗄️" ;;
        *docs*|*documentation*) echo "📚" ;;
        *test*) echo "🧪" ;;
        *)
            # Hash-based consistent emoji for unknown paths
            # Use a small set of emojis for consistency
            local emojis=("💼" "📁" "🛠️" "⚙️" "🔧" "📊" "🎯" "🚀" "💡" "🔬")
            local hash=$(echo -n "$path" | cksum | cut -d' ' -f1)
            local index=$((hash % ${#emojis[@]}))
            echo "${emojis[$index]}"
            ;;
    esac
}

workspace_emoji=$(get_workspace_emoji "$cwd")

# Extract model display name
model_name=$(echo "$input" | jq -r '.model.display_name')

# Extract session ID
session_id=$(echo "$input" | jq -r '.session_id')

# Read session name from cache (generated by hook)
session_name_cache="/tmp/claude-session-name-${session_id}.txt"
if [ -f "$session_name_cache" ]; then
    session_name=$(cat "$session_name_cache" 2>/dev/null || echo "")
fi

# Fallback if no cached name
if [ -z "$session_name" ]; then
    session_name=$(echo "$input" | jq -r '.session_name // "Unnamed Session"')
fi

# Read plan info from cache (generated by hook)
plan_cache="/tmp/claude-session-plan-${session_id}.txt"
plan_indicator=""
if [ -f "$plan_cache" ]; then
    plan_name=$(cat "$plan_cache" 2>/dev/null || echo "")
    if [ -n "$plan_name" ]; then
        plan_indicator=" 📋 $plan_name"
    fi
fi

# Read assistant action summary from cache (generated by PostToolUse hook)
assistant_summary_cache="/tmp/claude-assistant-summary-${session_id}.txt"
assistant_actions=""
if [ -f "$assistant_summary_cache" ]; then
    actions=$(cat "$assistant_summary_cache" 2>/dev/null || echo "")
    if [ -n "$actions" ]; then
        assistant_actions=" 🔧 $actions"
    fi
fi

# Calculate context window percentage
usage=$(echo "$input" | jq '.context_window.current_usage')
if [ "$usage" != "null" ]; then
    current=$(echo "$usage" | jq '.input_tokens + .cache_creation_input_tokens + .cache_read_input_tokens')
    size=$(echo "$input" | jq '.context_window.context_window_size')
    context_pct=$((current * 100 / size))
    context_display=" | ${context_pct}% ctx"
else
    context_display=""
fi

# Set up environment variables that Starship expects
export PWD="$cwd"
export STARSHIP_SHELL="bash"

# Change to the directory so git commands work correctly
cd "$cwd" 2>/dev/null || cd "$HOME"

# Run starship with the prompt command and strip the \[\] escape sequences
# These are bash readline escapes that aren't needed for status line display
# Also remove newlines to keep everything on a single line
starship_output=$(starship prompt 2>/dev/null | sed 's/\\\[//g; s/\\\]//g' | tr -d '\n')

# Generate session summary (line 2)
summary_cache="/tmp/claude-session-summary-${session_id}.txt"
summary_timestamp="/tmp/claude-session-summary-${session_id}.timestamp"
summary_msg_count="/tmp/claude-session-summary-${session_id}.msgcount"

# Check if we need to regenerate the summary
# Regenerate ONLY if:
# 1. Cache doesn't exist, OR
# 2. New messages exist since last check
# Idle sessions keep their cached summary indefinitely
should_regenerate=false
history_file="$HOME/.claude/history.jsonl"

# Get current message count
current_msg_count=0
if [ -f "$history_file" ]; then
    current_msg_count=$(grep -c "\"sessionId\":\"${session_id}\"" "$history_file" 2>/dev/null || echo "0")
    current_msg_count=$(echo "$current_msg_count" | tr -d '\n\r ')
fi
current_msg_count=${current_msg_count:-0}

if [ ! -f "$summary_cache" ]; then
    should_regenerate=true
else
    # Check if new messages exist
    last_msg_count=$(cat "$summary_msg_count" 2>/dev/null | tr -d '\n\r ' || echo "0")
    last_msg_count=${last_msg_count:-0}
    if [ "$current_msg_count" -gt "$last_msg_count" ]; then
        should_regenerate=true
    fi
fi

if [ "$should_regenerate" = true ]; then
    # Read from history.jsonl instead of transcript
    history_file="$HOME/.claude/history.jsonl"

    if [ -f "$history_file" ]; then
        # Process line-by-line to avoid jq -s parse errors
        first_user_msg=$(grep "\"sessionId\":\"${session_id}\"" "$history_file" | head -1 | jq -r '.display // ""')
        recent_user_msgs=$(grep "\"sessionId\":\"${session_id}\"" "$history_file" | tail -7 | jq -r '.display' | paste -sd ' ' -)
        user_msg_count=$(grep -c "\"sessionId\":\"${session_id}\"" "$history_file" 2>/dev/null || echo "0")
        user_msg_count=$(echo "$user_msg_count" | tr -d '\n\r ')
        user_msg_count=${user_msg_count:-0}
    else
        # No history file available
        first_user_msg=""
        recent_user_msgs=""
        user_msg_count=0
    fi

    # PRIMARY METHOD: AI summary with structured output
    # Use ALL user messages (no keyword filtering - let AI determine what's meaningful)
    all_user_msgs=$(grep "\"sessionId\":\"${session_id}\"" "$history_file" | jq -r '.display')

    # Get first + last 5 user messages for context
    first_user_msgs=$(echo "$all_user_msgs" | head -5 | paste -sd '. ' -)
    last_user_msgs=$(echo "$all_user_msgs" | tail -5 | paste -sd '. ' -)

    # Use structured JSON output with examples
    schema='{"type":"object","properties":{"task":{"type":"string","maxLength":60}},"required":["task"]}'
    prompt="Summarize this development session in max 60 chars based on what the user asked for and worked on.

User's requests (first 5): $first_user_msgs

User's requests (last 5): $last_user_msgs

Examples: Configuring statusline, Adding AI summaries, Fixing build errors, Refactoring auth module"

    # Call Claude with structured output using temp file for timeout handling
    ai_temp="/tmp/claude-ai-summary-${session_id}-$$.json"

    (
        echo "$prompt" | claude --model haiku -p --no-session-persistence --output-format json --json-schema "$schema" 2>/dev/null > "$ai_temp"
    ) &
    claude_pid=$!

    # Wait up to 3 seconds for completion (statusline needs to be fast)
    # First call will timeout and use fallback, subsequent calls benefit from cache
    for i in {1..3}; do
        if ! kill -0 $claude_pid 2>/dev/null; then
            break
        fi
        sleep 1
    done

    # Kill if still running (suppress all output)
    kill $claude_pid >/dev/null 2>&1
    wait $claude_pid >/dev/null 2>&1

    # Extract summary from temp file
    ai_summary=""
    if [ -f "$ai_temp" ] && [ -s "$ai_temp" ]; then
        ai_summary=$(jq -r '.structured_output.task // empty' "$ai_temp" 2>/dev/null)
        rm -f "$ai_temp"
    fi

    # Check if AI summary worked and is meaningful
    if [ -n "$ai_summary" ] && [ ${#ai_summary} -gt 10 ] && ! echo "$ai_summary" | grep -qiE "^(Ready|Awaiting|Clarify|Assist)"; then
        summary="$ai_summary"
    else
        # FALLBACK: Use first + last message
        first_line=$(echo "$first_user_msg" | head -n 1 | sed 's/^[[:space:]]*//;s/[[:space:]]*$//' | cut -c 1-47)
        last_user_msg=$(grep "\"sessionId\":\"${session_id}\"" "$history_file" | tail -1 | jq -r '.display // ""' | head -n 1 | sed 's/^[[:space:]]*//;s/[[:space:]]*$//' | cut -c 1-47)

        if [ "$user_msg_count" -le 1 ] || [ "$first_line" = "$last_user_msg" ]; then
            summary=$(echo "$first_line" | cut -c 1-100)
        else
            summary="${first_line} → ${last_user_msg}"
        fi
    fi

    # Clean up summary
    summary=$(echo "$summary" | tr -s ' ' | tr -d '\n\r' | sed 's/^[[:space:]]*//;s/[[:space:]]*$//')

    # Final fallback if still empty
    if [ -z "$summary" ]; then
        summary="Active session"
    fi

    # Cache the summary and message count
    echo "$summary" > "$summary_cache"
    date +%s > "$summary_timestamp"
    echo "$current_msg_count" > "$summary_msg_count"
else
    # Use cached summary
    summary=$(cat "$summary_cache" 2>/dev/null || echo "Active session")
fi

# Occasionally run shared cleanup (1% of the time to minimize overhead)
if [ $((RANDOM % 100)) -eq 0 ]; then
    bash "${CLAUDE_PLUGIN_ROOT}/hooks/scripts/cache-cleanup.sh" &
fi

# Combine workspace emoji, model name, starship output, and context percentage (line 1)
# Add session summary with session ID and assistant actions (line 2)
# Add session name with plan indicator (line 3)
printf "%s %s | %s%s\n%s%s | %s\n%s%s" "$workspace_emoji" "$model_name" "$starship_output" "$context_display" "$summary" "$assistant_actions" "$session_id" "$session_name" "$plan_indicator"
