# 3. Server-Side Performance

**Impact: HIGH**

Optimizing server-side rendering and data fetching eliminates server-side waterfalls and reduces response times.

## 3.1 Partial Pre-rendering (PPR)

**Impact: CRITICAL (sub-100ms LCP for dynamic pages)**

PPR allows a single route to have both static and dynamic parts. The static "shell" is served immediately from the edge, while dynamic "holes" (wrapped in Suspense) are streamed in as they finish.

**Pattern: Static Layout + Dynamic Content**

```tsx
// next.config.ts
export default {
  experimental: {
    ppr: 'incremental'
  }
}

// app/dashboard/page.tsx
import { Suspense } from 'react'
import { StaticShell, DynamicStats } from './components'

export const experimental_ppr = true // Enable for this route

export default function Page() {
  return (
    <main>
      <StaticShell /> {/* Served instantly as static HTML */}
      
      <Suspense fallback={<StatsSkeleton />}>
        <DynamicStats /> {/* Streamed in once data is ready */}
      </Suspense>
    </main>
  )
}
```

**Rule of Thumb**: Anything NOT wrapped in `<Suspense>` at the page level becomes part of the static pre-rendered shell.

## 3.2 Cross-Request LRU Caching

**Impact: HIGH (caches across requests)**

`React.cache()` only works within one request. For data shared across sequential requests (user clicks button A then button B), use an LRU cache.

**Implementation:**

```typescript
import { LRUCache } from 'lru-cache'

const cache = new LRUCache<string, any>({
  max: 1000,
  ttl: 5 * 60 * 1000  // 5 minutes
})

export async function getUser(id: string) {
  const cached = cache.get(id)
  if (cached) return cached

  const user = await db.user.findUnique({ where: { id } })
  cache.set(id, user)
  return user
}
```

Use when sequential user actions hit multiple endpoints needing the same data within seconds.

**With Vercel's [Fluid Compute](https://vercel.com/docs/fluid-compute):** LRU caching is especially effective because multiple concurrent requests can share the same function instance and cache. This means the cache persists across requests without needing external storage like Redis.

## 3.2 Minimize Serialization at RSC Boundaries

**Impact: HIGH (reduces data transfer size)**

The React Server/Client boundary serializes all object properties into strings and embeds them in the HTML response and subsequent RSC requests. This serialized data directly impacts page weight and load time, so **size matters a lot**. Only pass fields that the client actually uses.

**Incorrect: serializes all 50 fields**

```tsx
async function Page() {
  const user = await fetchUser()  // 50 fields
  return <Profile user={user} />
}

'use client'
function Profile({ user }: { user: User }) {
  return <div>{user.name}</div>  // uses 1 field
}
```

**Correct: serializes only 1 field**

```tsx
async function Page() {
  const user = await fetchUser()
  return <Profile name={user.name} />
}
```

## 3.3 Parallel Data Fetching with Component Composition

**Impact: CRITICAL (eliminates server-side waterfalls)**

React Server Components execute sequentially within a tree. Restructure with composition to parallelize data fetching.

**Incorrect: Sidebar waits for Page's fetch to complete**

```tsx
export default async function Page() {
  const header = await fetchHeader()
  return (
    <div>
      <div>{header}</div>
      <Sidebar />
    </div>
  )
}
```

**Correct: both fetch simultaneously**

```tsx
export default function Page() {
  return (
    <div>
      <Header />
      <Sidebar />
    </div>
  )
}
```

## 3.4 Per-Request Deduplication with React.cache()

**Impact: MEDIUM (deduplicates within request)**

Use `React.cache()` for server-side request deduplication. Authentication and database queries benefit most.

```typescript
import { cache } from 'react'

export const getCurrentUser = cache(async () => {
  const session = await auth()
  if (!session?.user?.id) return null
  return await db.user.findUnique({
    where: { id: session.user.id }
  })
})
```

## 3.5 Use Non-Blocking Operations for Side Effects

**Impact: MEDIUM (faster response times)**

Schedule work that should execute after a response is sent.

**Correct: using on-finished package**

```tsx
import onFinished from 'on-finished'

app.post('/api/action', async (req, res) => {
  await updateDatabase(req.body)

  onFinished(res, (err) => {
    if (err) return 
    logUserAction({ userAgent: req.headers['user-agent'], status: res.statusCode })
  })

  res.json({ status: 'success' })
})
```
