#!/bin/bash
# launch-agent.sh - Launch Claude Code in a new terminal for a worktree
#
# Usage: ./launch-agent.sh <worktree-path> [task-description] [--prompt "template"] [--headless]
#
# Prompt Modes:
#   --prompt "template"            Interactive mode (default): passes prompt as argument,
#                                  Claude starts with prompt pre-filled in input (cc "prompt")
#   --prompt "template" --headless Headless mode: uses -p flag, auto-executes immediately (cc -p "prompt")
#
# Examples:
#   ./launch-agent.sh ~/Projects/worktrees/my-project/feature-auth
#   ./launch-agent.sh ~/Projects/worktrees/my-project/feature-auth "Implement OAuth login"
#   ./launch-agent.sh ~/Projects/worktrees/my-project/feature-auth "" --prompt "/explore"
#   ./launch-agent.sh ~/Projects/worktrees/my-project/feature-auth "" --prompt "/review-code" --headless
#   ./launch-agent.sh ~/Projects/worktrees/my-project/feature-auth "Optimize" --prompt "analyze {{service}}"
#
# Template Variables (for --prompt):
#   {{service}}       - Branch slug (e.g., "feature-auth")
#   {{branch}}        - Full branch name (e.g., "feature/auth")
#   {{branch_slug}}   - Same as {{service}}
#   {{project}}       - Project name
#   {{worktree_path}} - Full worktree path
#   {{ports}}         - Allocated ports (comma-separated)
#   {{port}}          - First allocated port

set -e

WORKTREE_PATH="$1"
TASK="$2"
PROMPT_TEMPLATE=""
HEADLESS_MODE=false

# Parse optional arguments after positional args
shift 2 2>/dev/null || shift $# 2>/dev/null
while [[ $# -gt 0 ]]; do
    case "$1" in
        --prompt)
            PROMPT_TEMPLATE="$2"
            shift 2
            ;;
        --headless|-p)
            HEADLESS_MODE=true
            shift
            ;;
        *)
            shift
            ;;
    esac
done

# Validate input
if [ -z "$WORKTREE_PATH" ]; then
    echo "Error: Worktree path required"
    echo "Usage: $0 <worktree-path> [task-description]"
    exit 1
fi

# Find script directory and load config
SCRIPT_DIR="$(cd "$(dirname "$0")" && pwd)"
source "$SCRIPT_DIR/lib/config.sh"

# Load config values (user config -> template -> defaults)
TERMINAL=$(get_config "terminal" "ghostty")
SHELL_CMD=$(get_config "shell" "bash")
CLAUDE_CMD=$(get_config "claudeCommand" "claude")

# SEC-001: Security warning for dangerous mode
# The --dangerously-skip-permissions flag bypasses Claude's permission prompts,
# which grants full autonomous access. This is intentional for worktree automation
# but users should be aware of the security implications.
if [[ "$CLAUDE_CMD" == *"--dangerously-skip-permissions"* ]]; then
    echo "WARNING: Claude running in dangerous mode (--dangerously-skip-permissions)"
    echo "         This bypasses permission prompts and grants full autonomous access."
    echo "         To disable, set 'claudeCommand' in ~/.claude/worktree-manager.config.json"
    echo ""
fi

# Expand ~ in path
WORKTREE_PATH="${WORKTREE_PATH/#\~/$HOME}"

# Convert to absolute path if relative
if [[ "$WORKTREE_PATH" != /* ]]; then
    WORKTREE_PATH="$(pwd)/$WORKTREE_PATH"
fi

# Verify worktree exists
if [ ! -d "$WORKTREE_PATH" ]; then
    echo "Error: Worktree directory does not exist: $WORKTREE_PATH"
    exit 1
fi

# Verify it's a git worktree (has .git file or directory)
if [ ! -e "$WORKTREE_PATH/.git" ]; then
    echo "Error: Not a git worktree: $WORKTREE_PATH"
    exit 1
fi

# Get branch name
BRANCH=$(cd "$WORKTREE_PATH" && git branch --show-current 2>/dev/null)
if [ -z "$BRANCH" ]; then
    BRANCH=$(basename "$WORKTREE_PATH")
fi

# Get project name from path
PROJECT=$(basename "$(dirname "$WORKTREE_PATH")")

# Get branch slug (for template substitution)
BRANCH_SLUG=$(basename "$WORKTREE_PATH")

# Get ports from registry (for template substitution)
PORTS=""
PORT=""
if [ -f "$HOME/.claude/worktree-registry.json" ] && command -v jq &> /dev/null; then
    PORTS=$(jq -r ".worktrees[] | select(.worktreePath == \"$WORKTREE_PATH\") | .ports | join(\",\")" "$HOME/.claude/worktree-registry.json" 2>/dev/null || echo "")
    PORT=$(echo "$PORTS" | cut -d',' -f1)
fi

# SEC-MED-001: Template variable substitution function
# These variables are used in prompt strings passed to Claude, NOT in file operations.
# WORKTREE_PATH is validated above (exists, is git worktree) before substitution.
# Other variables (BRANCH, PROJECT, BRANCH_SLUG) are derived from git or path basenames.
# Path traversal is not a concern because outputs are text prompts, not file paths.
substitute_template() {
    local template="$1"
    template="${template//\{\{service\}\}/$BRANCH_SLUG}"
    template="${template//\{\{branch\}\}/$BRANCH}"
    template="${template//\{\{branch_slug\}\}/$BRANCH_SLUG}"
    template="${template//\{\{project\}\}/$PROJECT}"
    template="${template//\{\{worktree_path\}\}/$WORKTREE_PATH}"
    template="${template//\{\{ports\}\}/$PORTS}"
    template="${template//\{\{port\}\}/$PORT}"
    echo "$template"
}

# Build Claude command with prompt support
# - Headless mode (--headless): uses -p flag for auto-execution
# - Interactive mode (default): passes prompt as argument (pre-filled in input)
build_claude_cmd() {
    local base_cmd="$1"
    local prompt="$2"
    local headless="$3"

    if [ -n "$prompt" ]; then
        local substituted_prompt
        substituted_prompt=$(substitute_template "$prompt")
        # Escape single quotes for shell safety
        substituted_prompt="${substituted_prompt//\'/\'\\\'\'}"

        if [ "$headless" = "true" ]; then
            # Headless mode: -p flag auto-executes and exits
            echo "$base_cmd -p '$substituted_prompt'"
        else
            # Interactive mode: pass prompt as argument (pre-filled in Claude's input)
            echo "$base_cmd '$substituted_prompt'"
        fi
    else
        # No prompt: just launch claude
        echo "$base_cmd"
    fi
}

# Build final Claude command (with prompt if provided)
FINAL_CLAUDE_CMD=$(build_claude_cmd "$CLAUDE_CMD" "$PROMPT_TEMPLATE" "$HEADLESS_MODE")

# Build the command to run in the new terminal
# For fish: use 'and'/'or' instead of '&&'/'||'
if [ "$SHELL_CMD" = "fish" ]; then
    if [ -n "$TASK" ]; then
        INNER_CMD="cd '$WORKTREE_PATH'; and echo 'Worktree: $PROJECT / $BRANCH'; and echo 'Task: $TASK'; and echo ''; and $FINAL_CLAUDE_CMD"
    else
        INNER_CMD="cd '$WORKTREE_PATH'; and echo 'Worktree: $PROJECT / $BRANCH'; and echo ''; and $FINAL_CLAUDE_CMD"
    fi
else
    # bash/zsh syntax
    if [ -n "$TASK" ]; then
        INNER_CMD="cd '$WORKTREE_PATH' && echo 'Worktree: $PROJECT / $BRANCH' && echo 'Task: $TASK' && echo '' && $FINAL_CLAUDE_CMD"
    else
        INNER_CMD="cd '$WORKTREE_PATH' && echo 'Worktree: $PROJECT / $BRANCH' && echo '' && $FINAL_CLAUDE_CMD"
    fi
fi

# Launch based on terminal type
case "$TERMINAL" in
    ghostty)
        if ! command -v ghostty &> /dev/null && [ ! -d "/Applications/Ghostty.app" ]; then
            echo "Error: Ghostty not found"
            exit 1
        fi
        # Launch Ghostty with the command
        open -na "Ghostty.app" --args -e "$SHELL_CMD" -c "$INNER_CMD"
        ;;

    iterm2|iterm)
        # Create new WINDOW in iTerm2
        osascript <<EOF
tell application "iTerm2"
    create window with default profile
    tell current session of current window
        write text "cd '$WORKTREE_PATH' && $FINAL_CLAUDE_CMD"
    end tell
end tell
EOF
        ;;

    iterm2-tab|iterm-tab)
        # Create new TAB in current iTerm2 window
        osascript <<EOF
tell application "iTerm2"
    tell current window
        create tab with default profile
        tell current session
            write text "cd '$WORKTREE_PATH' && $FINAL_CLAUDE_CMD"
        end tell
    end tell
end tell
EOF
        ;;

    tmux)
        if ! command -v tmux &> /dev/null; then
            echo "Error: tmux not found"
            exit 1
        fi
        SESSION_NAME="wt-$PROJECT-$(echo "$BRANCH" | tr '/' '-')"
        tmux new-session -d -s "$SESSION_NAME" -c "$WORKTREE_PATH" "$SHELL_CMD -c '$FINAL_CLAUDE_CMD'"
        echo "   tmux session: $SESSION_NAME (attach with: tmux attach -t $SESSION_NAME)"
        ;;

    wezterm)
        if ! command -v wezterm &> /dev/null; then
            echo "Error: WezTerm not found"
            exit 1
        fi
        wezterm start --cwd "$WORKTREE_PATH" -- "$SHELL_CMD" -c "$INNER_CMD"
        ;;

    kitty)
        if ! command -v kitty &> /dev/null; then
            echo "Error: Kitty not found"
            exit 1
        fi
        kitty --detach --directory "$WORKTREE_PATH" "$SHELL_CMD" -c "$INNER_CMD"
        ;;

    alacritty)
        if ! command -v alacritty &> /dev/null; then
            echo "Error: Alacritty not found"
            exit 1
        fi
        alacritty --working-directory "$WORKTREE_PATH" -e "$SHELL_CMD" -c "$INNER_CMD" &
        ;;

    *)
        echo "Error: Unknown terminal type: $TERMINAL"
        echo "Supported: ghostty, iterm2, iterm2-tab, tmux, wezterm, kitty, alacritty"
        exit 1
        ;;
esac

echo "Launched Claude Code agent"
echo "   Terminal: $TERMINAL"
echo "   Project: $PROJECT"
echo "   Branch: $BRANCH"
echo "   Path: $WORKTREE_PATH"
if [ -n "$TASK" ]; then
    echo "   Task: $TASK"
fi
if [ -n "$PROMPT_TEMPLATE" ]; then
    if [ "$HEADLESS_MODE" = "true" ]; then
        echo "   Prompt: $(substitute_template "$PROMPT_TEMPLATE") (headless)"
    else
        echo "   Prompt: $(substitute_template "$PROMPT_TEMPLATE") (interactive)"
    fi
fi
